/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.util;

import java.lang.reflect.Field;

/**
 * Class that makes available code on Runtime's ClassUtils superior to the min mule version of the DB Client
 */
public final class ClassUtils {

  private ClassUtils() {}

  /**
   * Gets the field with the given fieldName of the TargetClass making it accessible.
   *
   * @param targetClass class to get the field from.
   * @param fieldName   the name of the field.
   * @param recursive   flag to lookup in subclasses or not
   * @return the Field object
   * @throws NoSuchFieldException when the request field does not exists.
   */
  public static Field getField(Class<?> targetClass, String fieldName, boolean recursive) throws NoSuchFieldException {
    Class<?> clazz = targetClass;
    Field field;

    while (!Object.class.equals(clazz)) {
      try {
        field = clazz.getDeclaredField(fieldName);
        return field;
      } catch (NoSuchFieldException e) {
        // ignore and look in superclass
        if (recursive) {
          clazz = clazz.getSuperclass();
        } else {
          break;
        }
      }
    }

    throw new NoSuchFieldException(String.format("Could not find field '%s' in class %s", fieldName, targetClass.getName()));
  }

  /**
   * Gets a field value of a given object
   *
   * @param target    object to get the field value from
   * @param fieldName the name of the field
   * @param recursive flag to lookup in subclasses
   * @param <T>       the expected type of the field.
   * @return the value of the field for the target object
   * @throws IllegalAccessException when the field is not reachable
   * @throws NoSuchFieldException   when the field does not exists
   */
  public static <T> T getFieldValue(Object target, String fieldName, boolean recursive)
      throws IllegalAccessException, NoSuchFieldException {
    Field f = getField(target.getClass(), fieldName, recursive);
    f.setAccessible(true);

    return (T) f.get(target);
  }

}
