/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal;

import static java.lang.String.format;
import static org.mule.runtime.core.api.event.EventContextFactory.create;
import static org.mule.runtime.dsl.api.component.config.DefaultComponentLocation.fromSingleComponent;

import org.mule.db.commons.internal.domain.connection.DbConnection;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionProvider;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.core.api.exception.NullExceptionHandler;
import org.mule.runtime.core.api.extension.ExtensionManager;
import org.mule.runtime.extension.api.runtime.config.ConfigurationInstance;

import java.sql.SQLException;
import java.util.List;

/**
 * Utility class for DB Functions
 *
 * @since 1.5.1, 1.6.0
 */
public class DbFunctionUtil {

  private final ExtensionManager extensionManager;

  private static final CoreEvent EVENT = CoreEvent.builder(create("DB", "dummy", fromSingleComponent("DB"),
                                                                  NullExceptionHandler.getInstance()))
      .message(Message.of("none"))
      .build();

  public DbFunctionUtil(ExtensionManager extensionManager) {
    this.extensionManager = extensionManager;
  }

  /**
   * Runs code with the proper connection.
   *
   * The connection is disconnected after the code is run.
   *
   * @param connectedFunction the code to run
   * @param connectionName    the database configuration to get the connection
   * @return the value returned by connectedFunction
   */
  public <T> T execute(ConnectedFunction<T> connectedFunction, String connectionName) throws ConnectionException, SQLException {
    ConnectionProvider<Object> connectionProvider = getConnectionProvider(connectionName);
    DbConnection dbConnection = null;
    try {
      Object connection = connectionProvider.connect();

      if (!(connection instanceof DbConnection)) {
        throw new ConnectionException("Connection is not a DB Connection");
      }
      dbConnection = (DbConnection) connection;

      return connectedFunction.apply(dbConnection);
    } finally {
      if (dbConnection != null)
        connectionProvider.disconnect(dbConnection);
    }
  }

  @SuppressWarnings("unchecked")
  private <T> ConnectionProvider<T> getConnectionProvider(String configName) {
    ConfigurationInstance configuration = extensionManager.getConfiguration(configName, EVENT);
    return configuration.getConnectionProvider()
        .orElseThrow(() -> new RuntimeException(format("Unable to obtain a connection for configuration: [%s]", configName)));
  }

  /**
   * Interface for {@link #execute(ConnectedFunction, String)}.
   */
  @FunctionalInterface
  public interface ConnectedFunction<T> {

    T apply(DbConnection connection) throws ConnectionException, SQLException;
  }

  /**
   * @deprecated Use {@link #execute(ConnectedFunction, String)} instead.
   */
  @Deprecated
  public Object execute(WithConnection withConnection, List<Object> values, String typeName, String connectionName) {
    try {
      return execute(connection -> withConnection.execute(connection, values, typeName), connectionName);
    } catch (Throwable t) {
      throw new RuntimeException("An error occurred when trying to create JDBC Structure. " + t.getMessage(), t);
    }
  }

  /**
   * Interface for {@link #execute(WithConnection, List, String, String)}.
   *
   * @deprecated Use {@link #execute(ConnectedFunction, String)} instead.
   */
  @Deprecated
  @FunctionalInterface
  public interface WithConnection {

    Object execute(DbConnection connection, List<Object> values, String typeName) throws ConnectionException, SQLException;
  }
}
