/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.domain.type;

import static java.lang.String.format;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;

import org.mule.db.commons.internal.domain.connection.DbConnection;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.core.api.util.IOUtils;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringReader;
import java.io.Writer;

import java.sql.Clob;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.SQLException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Defines a data type for {@link Clob}
 */
public class ClobResolvedDataType extends ResolvedDbType {

  private static final Logger LOGGER = LoggerFactory.getLogger(ClobResolvedDataType.class);
  private static final String JTDS_DRIVER = "jTDS";

  public ClobResolvedDataType(int id, String name) {
    super(id, name);
  }

  @Override
  public void setParameterValue(PreparedStatement statement, int index, Object value, DbConnection connection)
      throws SQLException {
    if (value != null && !(value instanceof Clob)) {
      try {
        LOGGER.debug("Creating CLOB object");

        if (value instanceof String) {
          handleIfString(statement, index, value, connection);
        } else if (value instanceof ByteArrayInputStream) {
          handleIfByteArrayInputStream(statement, index, value, connection);
        } else {
          handleIfInputStreamOrFail(statement, index, value, connection);
        }
      } catch (SQLException sqlException) {
        LOGGER.debug("Error creating CLOB object. Using alternative way to set CLOB object.", sqlException);
        handlePriorVersions(statement, index, value);
      } catch (IOException ioException) {
        throw new MuleRuntimeException(createStaticMessage("Cannot consume content of CLOB from a value of type '%s'",
                                                           value.getClass()),
                                       ioException);
      }
    } else {
      super.setParameterValue(statement, index, value, connection);
    }
  }

  private void handleIfString(PreparedStatement statement, int index, Object value, DbConnection connection) throws SQLException {
    Clob clob = statement.getConnection().createClob();
    clob.setString(1, (String) value);

    super.setParameterValue(statement, index, clob, connection);
  }

  private void handleIfByteArrayInputStream(PreparedStatement statement, int index, Object value, DbConnection connection)
      throws SQLException, IOException {
    Clob clob = statement.getConnection().createClob();
    OutputStream outputStream = clob.setAsciiStream(1);

    outputStream.write(IOUtils.toByteArray((InputStream) value));
    outputStream.flush();
    outputStream.close();

    super.setParameterValue(statement, index, clob, connection);
  }

  private void handleIfInputStreamOrFail(PreparedStatement statement, int index, Object value, DbConnection connection)
      throws SQLException, IOException {
    Reader reader;

    if (value instanceof InputStream) {
      reader = new InputStreamReader((InputStream) value);
    } else if (value instanceof TypedValue<?> && ((TypedValue<?>) value).getValue() instanceof InputStream) {
      reader = new InputStreamReader((InputStream) ((TypedValue<?>) value).getValue());
    } else {
      throw new IllegalArgumentException(format("Cannot create a CLOB from a value of type '%s'", value.getClass()));
    }

    Clob clob = statement.getConnection().createClob();
    Writer writer = clob.setCharacterStream(1);
    IOUtils.copyLarge(reader, writer);
    writer.flush();
    writer.close();
    super.setParameterValue(statement, index, clob, connection);
  }

  /**
   * createClob method has been add to JDBC API in version 3.0. Since we have to support any driver that works with JDK 1.8 we try
   * an alternative way to set CLOB objects.
   */
  private void handlePriorVersions(PreparedStatement statement, int index, Object value) throws SQLException {
    DatabaseMetaData metaData = statement.getConnection().getMetaData();
    String valueString;

    // We do a best-effort
    if (value instanceof String) {
      valueString = (String) value;
    } else if (value instanceof InputStream) {
      valueString = IOUtils.toString((InputStream) value);
    } else {
      throw new IllegalArgumentException(format("Cannot create a CLOB from a value of type '%s'", value.getClass()));
    }

    // We handle the possibility of dealing with a jTDS driver
    if (metaData != null && metaData.getDriverName() != null && metaData.getDriverName().contains(JTDS_DRIVER)) {
      statement.setString(index, valueString);
    } else {
      statement.setCharacterStream(index, new StringReader(valueString), valueString.length());
    }
  }

}
