/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.parser;

import static org.mule.runtime.core.api.util.StringUtils.isEmpty;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.db.commons.internal.domain.param.DefaultInputQueryParam;
import org.mule.db.commons.internal.domain.param.QueryParam;
import org.mule.db.commons.internal.domain.query.QueryTemplate;
import org.mule.db.commons.internal.domain.query.QueryType;
import org.mule.db.commons.internal.domain.type.UnknownDbType;

import java.util.ArrayList;
import java.util.List;

import org.mule.db.commons.internal.parser.statement.detector.StatementTypeDetector;
import org.mule.db.commons.internal.parser.statement.detector.UnknownStatementTypeException;
import org.slf4j.Logger;


/**
 * Simple SQL parser
 */
public class SimpleQueryTemplateParser implements QueryTemplateParser {

  private static final Logger LOGGER = getLogger(SimpleQueryTemplateParser.class);
  /*
   * TODO Keeping for backwards compatibility. Please check
   * https://github.com/mulesoft/mule-db-client/blob/4c974d8f3815fb153b885adee0f1aa24bf9441f4/src/main/java/org/mule/db/commons/
   * internal/parser/SimpleQueryTemplateParser.java#L100
   */
  private static final QueryType DEFAULT_QUERY_TYPE = QueryType.DDL;

  private static final char[] PARAMETER_SEPARATORS =
      new char[] {'"', '\'', ':', '&', ',', ';', '(', ')', '|', '=', '+', '-', '*', '%', '/', '\\', '<', '>', '^'};

  /**
   * Set of characters that qualify as comment or quotes starting characters.
   */
  private static final String[] BEGIN_SKIP =
      new String[] {"'", "\"", "--", "/*"};

  /**
   * Set of characters that at are the corresponding comment or quotes ending characters.
   */
  private static final String[] END_SKIP =
      new String[] {"'", "\"", "\n", "*/"};

  private static final char COLON_ESCAPE_CHARACTER = '\\';

  private final StatementTypeDetector queryTypeDetector = new StatementTypeDetector();


  @Override
  public QueryTemplate parse(String sql) {
    QueryType queryType;
    sql = sql.trim();

    LOGGER.debug("Parsing SQL: {}", sql);
    if (isEmpty(sql)) {
      throw new QueryTemplateParsingException("SQL text cannot be empty");
    }

    try {
      queryType = queryTypeDetector.detect(sql);
    } catch (UnknownStatementTypeException ex) {
      LOGGER.debug("Failed to detect query Type for {}. Falling back to default", sql, ex);
      queryType = DEFAULT_QUERY_TYPE;
    }
    return doParse(sql, queryType);
  }


  private QueryTemplate doParse(String sqlText, QueryType queryType) {

    StringBuilder sqlToUse = new StringBuilder();
    List<QueryParam> parameterList = new ArrayList<>();
    char[] sqlTextChars = sqlText.toCharArray();
    int tokenStart = 0;
    int paramIndex = 1;

    while (tokenStart < sqlTextChars.length) {
      int skipToPosition;

      while (tokenStart < sqlTextChars.length) {

        skipToPosition = skipCommentsAndQuotes(sqlTextChars, tokenStart);
        if (tokenStart == skipToPosition) {
          break;
        } else {
          sqlToUse = sqlToUse.append(sqlText, tokenStart, skipToPosition);
          tokenStart = skipToPosition;
        }
      }
      if (tokenStart >= sqlTextChars.length) {
        break;
      }
      char currentChar = sqlTextChars[tokenStart];
      int tokenEnd = tokenStart + 1;
      if (tokenEnd < sqlTextChars.length && currentChar == '#' && sqlTextChars[tokenEnd] == '[') {
        int openBrackets = 0;
        while (tokenEnd < sqlTextChars.length) {
          if (sqlTextChars[tokenEnd] == ']') {
            openBrackets--;
          } else if (sqlTextChars[tokenEnd] == '[') {
            openBrackets++;
          }

          if (openBrackets == 0) {
            break;
          }
          tokenEnd++;

        }
        if (tokenEnd == sqlTextChars.length) {
          throw new QueryTemplateParsingException("Invalid Mule expression: " + sqlText.substring(tokenStart));
        }

        tokenEnd++;
        String value = sqlText.substring(tokenStart, tokenEnd);
        QueryParam inputParam = new DefaultInputQueryParam(paramIndex++, UnknownDbType.getInstance(), value);
        parameterList.add(inputParam);
        sqlToUse = sqlToUse.append("?");
        tokenStart = tokenEnd;
      } else if (currentChar == COLON_ESCAPE_CHARACTER && tokenEnd < sqlTextChars.length && ':' == sqlTextChars[tokenEnd]) {
        sqlToUse = sqlToUse.append(':');
        tokenEnd++;
        tokenStart = tokenEnd;
      } else if (currentChar == ':') {
        if (tokenEnd < sqlTextChars.length && '=' == sqlTextChars[tokenEnd]) {
          sqlToUse = sqlToUse.append(currentChar);
          tokenStart++;
        } else if (tokenEnd < sqlTextChars.length && ':' == sqlTextChars[tokenEnd]) {
          sqlToUse = sqlToUse.append(currentChar).append(sqlTextChars[tokenEnd]);
          tokenStart = tokenStart + 2;
        } else {
          String parameter;

          while (tokenEnd < sqlTextChars.length && !isParameterSeparator(sqlTextChars[tokenEnd])) {
            tokenEnd++;
          }
          if (tokenEnd - tokenStart > 1) {
            sqlToUse = sqlToUse.append("?");
            parameter = sqlText.substring(tokenStart + 1, tokenEnd);
            QueryParam inputParam = new DefaultInputQueryParam(paramIndex++, UnknownDbType.getInstance(), null, parameter);
            parameterList.add(inputParam);
          }
          tokenStart = tokenEnd;
        }
      } else if (isParamChar(currentChar)) {
        QueryParam inputParam = new DefaultInputQueryParam(paramIndex++, UnknownDbType.getInstance(), null);
        parameterList.add(inputParam);
        tokenStart++;
        sqlToUse = sqlToUse.append(currentChar);
      } else {
        sqlToUse = sqlToUse.append(currentChar);
        tokenStart++;
      }
    }
    return new QueryTemplate(sqlToUse.toString(), queryType, parameterList);
  }

  private boolean isParamChar(char c) {
    return c == '?';
  }


  private static boolean isParameterSeparator(char c) {
    if (Character.isWhitespace(c)) {
      return true;
    }
    for (char separator : PARAMETER_SEPARATORS) {
      if (c == separator) {
        return true;
      }
    }
    return false;
  }

  private static int skipCommentsAndQuotes(char[] statement, int position) {

    for (int i = 0; i < BEGIN_SKIP.length; i++) {
      if (statement[position] == BEGIN_SKIP[i].charAt(0)) {
        boolean match = true;
        for (int j = 1; j < BEGIN_SKIP[i].length(); j++) {
          if (!(statement[position + j] == BEGIN_SKIP[i].charAt(j))) {
            match = false;
            break;
          }
        }
        if (match) {
          int offset = BEGIN_SKIP[i].length();
          for (int m = position + offset; m < statement.length; m++) {
            if (statement[m] == END_SKIP[i].charAt(0)) {
              boolean endMatch = true;
              int endPos = m;
              for (int n = 1; n < END_SKIP[i].length(); n++) {
                if (m + n >= statement.length) {
                  // last comment not closed properly
                  return statement.length;
                }
                if (!(statement[m + n] == END_SKIP[i].charAt(n))) {
                  endMatch = false;
                  break;
                }
                endPos = m + n;
              }
              if (endMatch) {
                // found character sequence ending comment or quote
                return endPos + 1;
              }
            }
          }
          // character sequence ending comment or quote not found
          return statement.length;
        }

      }
    }
    return position;
  }

}
