/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.operation;

import org.mule.db.commons.AbstractDbConnector;
import org.mule.db.commons.api.param.BulkQueryDefinition;
import org.mule.db.commons.api.param.BulkScript;
import org.mule.db.commons.api.param.ParameterizedStatementDefinition;
import org.mule.db.commons.api.param.QuerySettings;
import org.mule.db.commons.internal.domain.connection.DbConnection;
import org.mule.db.commons.internal.domain.executor.BulkExecutor;
import org.mule.db.commons.internal.domain.executor.BulkUpdateExecutor;
import org.mule.db.commons.internal.domain.executor.QueryExecutor;
import org.mule.db.commons.internal.domain.query.BulkQuery;
import org.mule.db.commons.internal.domain.query.Query;
import org.mule.db.commons.internal.domain.query.QueryParamValue;
import org.mule.db.commons.internal.domain.query.QueryType;
import org.mule.db.commons.internal.domain.statement.ConfigurableStatementFactory;
import org.mule.db.commons.internal.parser.QueryTemplateParser;
import org.mule.db.commons.internal.parser.SimpleQueryTemplateParser;
import org.mule.db.commons.internal.resolver.query.BulkQueryFactory;
import org.mule.db.commons.internal.resolver.query.BulkQueryResolver;
import org.mule.db.commons.internal.resolver.query.DefaultBulkQueryFactory;
import org.mule.db.commons.internal.resolver.query.FileBulkQueryFactory;
import org.mule.db.commons.internal.resolver.query.QueryResolver;
import org.mule.db.commons.internal.util.DefaultFileReader;

import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import static java.util.Arrays.asList;
import static java.util.Optional.*;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.mule.db.commons.internal.domain.query.QueryType.DELETE;
import static org.mule.db.commons.internal.domain.query.QueryType.INSERT;
import static org.mule.db.commons.internal.domain.query.QueryType.MERGE;
import static org.mule.db.commons.internal.domain.query.QueryType.STORE_PROCEDURE_CALL;
import static org.mule.db.commons.internal.domain.query.QueryType.TRUNCATE;
import static org.mule.db.commons.internal.domain.query.QueryType.UPDATE;
import static org.mule.runtime.api.metadata.TypedValue.unwrap;

/**
 * Contains a set of operations for performing bulk DML operations from a single statement.
 * 
 * @since 1.0
 */
public class BulkOperations extends BaseDbOperations {

  protected BulkQueryResolver bulkQueryResolver;
  protected QueryTemplateParser queryTemplateParser;
  protected Function<ConfigurableStatementFactory, BulkExecutor> bulkExecutorSupplier;

  private BulkOperations(QueryResolver<ParameterizedStatementDefinition> queryResolver,
                         ConfigurableStatementFactory statementFactory, BulkQueryResolver bulkQueryResolver,
                         QueryTemplateParser queryTemplateParser,
                         Function<ConfigurableStatementFactory, BulkExecutor> bulkExecutor,
                         Function<ConfigurableStatementFactory, QueryExecutor> updateExecutor) {
    super(queryResolver, statementFactory, updateExecutor);
    this.bulkQueryResolver = bulkQueryResolver;
    this.queryTemplateParser = queryTemplateParser;
    this.bulkExecutorSupplier = bulkExecutor;
  }

  protected static BulkQueryResolver getDefaultBulkQueryResolver() {
    return new BulkQueryResolver();
  }

  protected static QueryTemplateParser getDefaultQueryTemplateParser() {
    return new SimpleQueryTemplateParser();
  }

  protected static Function<ConfigurableStatementFactory, BulkExecutor> getDefaultBulkExecutor() {
    return BulkUpdateExecutor::new;
  }

  /**
   * Allows executing one insert statement various times using different parameter bindings. This happens using one single
   * Database statement, which has performance advantages compared to executing one single update operation various times.
   *
   * @param query               a {@link BulkQueryDefinition} as a parameter group
   * @param bulkInputParameters A {@link List} of {@link Map}s in which every list item represents a row to be inserted, and the
   *                            map contains the parameter names as keys and the value the parameter is bound to.
   * @param connector           the acting connector
   * @param connection          the acting connection
   * @return an array of update counts containing one element for each executed command. The elements of the array are ordered
   *         according to the order in which commands were added to the batch.
   * @throws SQLException if an error is produced
   */
  public int[] bulkInsert(List<Map<String, Object>> bulkInputParameters,
                          BulkQueryDefinition query,
                          AbstractDbConnector connector,
                          DbConnection connection,
                          StreamingHelper streamingHelper)
      throws SQLException {

    return singleQueryBulk(query, bulkInputParameters, connector, connection, streamingHelper, INSERT);
  }

  /**
   * Allows executing one update statement various times using different parameter bindings. This happens using one single
   * Database statement, which has performance advantages compared to executing one single update operation various times.
   *
   * @param query               a {@link BulkQueryDefinition} as a parameter group
   * @param bulkInputParameters A {@link List} of {@link Map}s in which every list item represents a row to be inserted, and the
   *                            map contains the parameter names as keys and the value the parameter is bound to.
   * @param connector           the acting connector
   * @param connection          the acting connection
   * @return an array of update counts containing one element for each executed command. The elements of the array are ordered
   *         according to the order in which commands were added to the batch.
   * @throws SQLException if an error is produced
   */
  public int[] bulkUpdate(List<Map<String, Object>> bulkInputParameters,
                          BulkQueryDefinition query,
                          AbstractDbConnector connector,
                          DbConnection connection,
                          StreamingHelper streamingHelper)
      throws SQLException {

    return singleQueryBulk(query, bulkInputParameters, connector, connection, streamingHelper, UPDATE, TRUNCATE, MERGE,
                           STORE_PROCEDURE_CALL);
  }

  /**
   * Allows executing one delete statement various times using different parameter bindings. This happens using one single
   * Database statement, which has performance advantages compared to executing one single delete operation various times.
   *
   * @param query               a {@link BulkQueryDefinition} as a parameter group
   * @param bulkInputParameters A {@link List} of {@link Map}s in which every list item represents a row to be inserted, and the
   *                            map contains the parameter names as keys and the value the parameter is bound to.
   * @param connector           the acting connector
   * @param connection          the acting connection
   * @return an array of update counts containing one element for each executed command. The elements of the array are ordered
   *         according to the order in which commands were added to the batch.
   * @throws SQLException if an error is produced
   */

  public int[] bulkDelete(List<Map<String, Object>> bulkInputParameters,
                          BulkQueryDefinition query,
                          AbstractDbConnector connector,
                          DbConnection connection,
                          StreamingHelper streamingHelper)
      throws SQLException {

    return singleQueryBulk(query, bulkInputParameters, connector, connection, streamingHelper, DELETE);
  }

  /**
   * Executes a SQL script in one single Database statement. The script is executed as provided by the user, without any parameter
   * binding.
   *
   * @param script     a {@link BulkScript} as a parameter group
   * @param settings   a {@link QuerySettings} as a parameter group
   * @param connection the acting connection
   * @return an array of update counts containing one element for each executed command. The elements of the array are ordered
   *         according to the order in which commands were added to the batch.
   * @throws SQLException if an error is produced
   */
  public int[] executeScript(BulkScript script,
                             QuerySettings settings,
                             DbConnection connection)
      throws SQLException {

    BulkQueryFactory bulkQueryFactory;

    if (!isEmpty(script.getFile())) {
      bulkQueryFactory = new FileBulkQueryFactory(script.getFile(), queryTemplateParser, new DefaultFileReader());
    } else {
      bulkQueryFactory = new DefaultBulkQueryFactory(queryTemplateParser, script.getSql());
    }

    BulkQuery bulkQuery = bulkQueryFactory.resolve();

    BulkExecutor bulkExecutor = bulkExecutorSupplier.apply(getStatementFactory(settings));

    return (int[]) bulkExecutor.execute(connection, bulkQuery);
  }


  private int[] singleQueryBulk(BulkQueryDefinition query,
                                List<Map<String, Object>> values,
                                AbstractDbConnector connector,
                                DbConnection connection,
                                StreamingHelper streamingHelper,
                                QueryType... queryType)
      throws SQLException {

    final Query resolvedQuery = resolveQuery(query, connector, connection, streamingHelper, queryType);

    List<List<QueryParamValue>> paramSets = resolveParamSets(values);

    BulkExecutor bulkExecutor = bulkExecutorSupplier.apply(getStatementFactory(query));

    return (int[]) bulkExecutor.execute(connection, resolvedQuery, paramSets);
  }

  protected Query resolveQuery(BulkQueryDefinition query,
                               AbstractDbConnector connector,
                               DbConnection connection,
                               StreamingHelper streamingHelper,
                               QueryType... validTypes) {
    final Query resolvedQuery =
        bulkQueryResolver.resolve(query, connector, connection, streamingHelper, connection.getCacheTemplates());
    validateQueryType(resolvedQuery.getQueryTemplate(), asList(validTypes));
    validateNoParameterTypeIsUnused(resolvedQuery, query.getParameterTypes());
    return resolvedQuery;
  }

  private List<List<QueryParamValue>> resolveParamSets(List<Map<String, Object>> values) {
    List<List<QueryParamValue>> parameterSet = new ArrayList<>();
    for (Object value : values) {
      Map<String, Object> map = unwrap(value);
      parameterSet
          .add(map.entrySet().stream().map(entry -> new QueryParamValue(entry.getKey(), entry.getValue())).collect(toList()));
    }
    return parameterSet;
  }

  public static class Builder {

    private Optional<QueryResolver<ParameterizedStatementDefinition>> queryResolverOptional = Optional.empty();
    private Optional<BulkQueryResolver> bulkQueryResolverOptional = Optional.empty();
    private Optional<QueryTemplateParser> queryTemplateParserOptional = Optional.empty();
    private Optional<ConfigurableStatementFactory> statementFactoryOptional = Optional.empty();
    private Optional<Function<ConfigurableStatementFactory, BulkExecutor>> bulkExecutorOptional = Optional.empty();
    private Optional<Function<ConfigurableStatementFactory, QueryExecutor>> updateExecutorOptional = Optional.empty();

    public BulkOperations.Builder withQueryResolver(QueryResolver<ParameterizedStatementDefinition> queryResolver) {
      this.queryResolverOptional = of(queryResolver);
      return this;
    }

    public BulkOperations.Builder withBulkQueryResolver(BulkQueryResolver bulkQueryResolver) {
      this.bulkQueryResolverOptional = of(bulkQueryResolver);
      return this;
    }

    public BulkOperations.Builder withQueryParser(QueryTemplateParser queryTemplateParser) {
      this.queryTemplateParserOptional = of(queryTemplateParser);
      return this;
    }

    public BulkOperations.Builder withStatementFactory(ConfigurableStatementFactory statementFactory) {
      this.statementFactoryOptional = of(statementFactory);
      return this;
    }

    public BulkOperations.Builder withBulkUpdateExecutor(Function<ConfigurableStatementFactory, BulkExecutor> bulkExecutor) {
      this.bulkExecutorOptional = of(bulkExecutor);
      return this;
    }

    public BulkOperations.Builder withUpdateExecutor(Function<ConfigurableStatementFactory, QueryExecutor> updateExecutor) {
      this.updateExecutorOptional = of(updateExecutor);
      return this;
    }

    public BulkOperations build() {
      return new BulkOperations(queryResolverOptional.orElse(getDefaultQueryResolver()),
                                statementFactoryOptional.orElse(getDefaultStatementFactory()),
                                bulkQueryResolverOptional.orElse(getDefaultBulkQueryResolver()),
                                queryTemplateParserOptional.orElse(getDefaultQueryTemplateParser()),
                                bulkExecutorOptional.orElse(getDefaultBulkExecutor()),
                                updateExecutorOptional.orElse(getDefaultUpdateExecutor()));
    }

  }

}
