/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.resolver.query;

import com.github.benmanes.caffeine.cache.Cache;
import org.mule.db.commons.AbstractDbConnector;
import org.mule.db.commons.api.param.ParameterType;
import org.mule.db.commons.api.param.StatementDefinition;
import org.mule.db.commons.internal.domain.connection.DbConnection;
import org.mule.db.commons.internal.domain.param.DefaultInOutQueryParam;
import org.mule.db.commons.internal.domain.param.DefaultInputQueryParam;
import org.mule.db.commons.internal.domain.param.DefaultOutputQueryParam;
import org.mule.db.commons.internal.domain.param.InOutQueryParam;
import org.mule.db.commons.internal.domain.param.InputQueryParam;
import org.mule.db.commons.internal.domain.param.OutputQueryParam;
import org.mule.db.commons.internal.domain.param.QueryParam;
import org.mule.db.commons.internal.domain.query.Query;
import org.mule.db.commons.internal.domain.query.QueryParamValue;
import org.mule.db.commons.internal.domain.query.QueryTemplate;
import org.mule.db.commons.internal.domain.type.CompositeDbTypeManager;
import org.mule.db.commons.internal.domain.type.DbType;
import org.mule.db.commons.internal.domain.type.DbTypeManager;
import org.mule.db.commons.internal.domain.type.DynamicDbType;
import org.mule.db.commons.internal.domain.type.StaticDbTypeManager;
import org.mule.db.commons.internal.domain.type.UnknownDbType;
import org.mule.db.commons.internal.parser.QueryTemplateParser;
import org.mule.db.commons.internal.parser.SimpleQueryTemplateParser;
import org.mule.db.commons.internal.resolver.param.GenericParamTypeResolverFactory;
import org.mule.db.commons.internal.resolver.param.ParamTypeResolverFactory;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.i18n.I18nMessageFactory;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import static org.apache.commons.collections.CollectionUtils.isEmpty;
import static org.apache.commons.lang3.StringUtils.isBlank;

public abstract class AbstractQueryResolver<T extends StatementDefinition<?>> implements QueryResolver<T> {

  private QueryTemplateParser queryTemplateParser = new SimpleQueryTemplateParser();

  @Override
  public Query resolve(T statementDefinition, AbstractDbConnector connector, DbConnection connection,
                       StreamingHelper streamingHelper, Cache<String, QueryTemplate> queryTemplates) {
    QueryTemplate queryTemplate = getQueryTemplate(connector, connection, statementDefinition, queryTemplates);
    return new Query(queryTemplate, resolveParams(statementDefinition, queryTemplate, streamingHelper));
  }

  protected abstract List<QueryParamValue> resolveParams(T statementDefinition, QueryTemplate template,
                                                         StreamingHelper streamingHelper);

  protected QueryTemplate createQueryTemplate(T statementDefinition, AbstractDbConnector connector, DbConnection connection) {
    if (isBlank(statementDefinition.getSql())) {
      throw new IllegalArgumentException("Statement doesn't contain a SQL query. Please provide one or reference a template which does");
    }

    QueryTemplate queryTemplate = queryTemplateParser.parse(statementDefinition.getSql());
    if (needsParamTypeResolution(queryTemplate)) {
      List<ParameterType> parameterTypes = statementDefinition.getParameterTypes();
      Map<Integer, DbType> paramTypes = getParameterTypes(connector, connection, queryTemplate, parameterTypes);
      queryTemplate = resolveQueryTemplate(queryTemplate, paramTypes);
    }

    return queryTemplate;
  }

  private Map<Integer, DbType> getParameterTypes(AbstractDbConnector connector, DbConnection connection,
                                                 QueryTemplate queryTemplate,
                                                 List<ParameterType> types) {
    ParamTypeResolverFactory paramTypeResolverFactory =
        new GenericParamTypeResolverFactory(createTypeManager(connector, connection));

    try {
      return paramTypeResolverFactory.create(queryTemplate).getParameterTypes(connection, queryTemplate, types);
    } catch (SQLException e) {
      throw new QueryResolutionException("Cannot resolve parameter types", e);
    }
  }

  private QueryTemplate getQueryTemplate(AbstractDbConnector connector, DbConnection connection, T statementDefinition,
                                         Cache<String, QueryTemplate> queryTemplates) {
    try {
      return queryTemplates.get(statementDefinition.getSql(),
                                key -> createQueryTemplate(statementDefinition, connector, connection));
    } catch (ModuleException e) {
      throw e;
    } catch (Exception e) {
      if (e.getCause() instanceof ModuleException) {
        throw (ModuleException) e.getCause();
      }

      throw new MuleRuntimeException(I18nMessageFactory
          .createStaticMessage("Could not resolve query: " + statementDefinition.getSql()), e);
    }
  }

  private QueryTemplate resolveQueryTemplate(QueryTemplate queryTemplate, Map<Integer, DbType> paramTypes) {
    List<QueryParam> newParams = new ArrayList<>();

    for (QueryParam originalParam : queryTemplate.getParams()) {
      DbType type = paramTypes.get(originalParam.getIndex());
      QueryParam newParam;

      if (type == null) {
        throw new IllegalArgumentException("Unknown parameter type of " + originalParam.getName());
      }

      if (originalParam instanceof InOutQueryParam) {
        newParam = new DefaultInOutQueryParam(originalParam.getIndex(), type, originalParam.getName(),
                                              ((InOutQueryParam) originalParam).getValue());
      } else if (originalParam instanceof InputQueryParam) {
        newParam =
            new DefaultInputQueryParam(originalParam.getIndex(), type, ((InputQueryParam) originalParam).getValue(),
                                       originalParam.getName());
      } else if (originalParam instanceof OutputQueryParam) {
        newParam = new DefaultOutputQueryParam(originalParam.getIndex(), type, originalParam.getName());
      } else {
        throw new IllegalArgumentException("Unknown parameter type: " + originalParam.getClass().getName());
      }

      newParams.add(newParam);
    }

    return new QueryTemplate(queryTemplate.getSqlText(), queryTemplate.getType(), newParams);
  }

  private boolean needsParamTypeResolution(QueryTemplate template) {
    return template.getParams().stream()
        .map(QueryParam::getType)
        .anyMatch(type -> type == UnknownDbType.getInstance() || type instanceof DynamicDbType);
  }

  protected DbTypeManager createTypeManager(AbstractDbConnector connector, DbConnection connection) {
    List<DbTypeManager> typeManagers = new LinkedList<>();
    List<DbTypeManager> vendorTypeManagers = new LinkedList<>();
    typeManagers.add(connector.getTypeManager());

    collectTypeManager(vendorTypeManagers, connection.getVendorDataTypes());
    collectTypeManager(vendorTypeManagers, connection.getCustomDataTypes());

    return new CompositeDbTypeManager(vendorTypeManagers, typeManagers);
  }

  private void collectTypeManager(List<DbTypeManager> collector, List<DbType> extraDataTypes) {
    if (!isEmpty(extraDataTypes)) {
      collector.add(new StaticDbTypeManager(extraDataTypes));
    }
  }

}
