/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.domain.connection.enhydra.wrappers;

import org.enhydra.jdbc.standard.StandardDataSource;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.Connection;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Properties;

import static org.mule.db.commons.internal.util.ClassUtils.getFieldValue;

/**
 * Utility class that allows us to modify Enhydra's behavior by overriding its methods.
 *
 * @since 1.3.0
 */
public final class EnhydraWrapperUtils {

  private static final String SSL_KEYSTORE = "javax.net.ssl.keyStore";
  private static final String SSL_KEYSTORE_TYPE = "javax.net.ssl.keyStoreType";
  private static final String SSL_KEYSTORE_PASSWORD = "javax.net.ssl.keyStorePassword";
  private static final String SSL_TRUSTSTORE = "javax.net.ssl.trustStore";
  private static final String SSL_TRUSTSTORE_TYPE = "javax.net.ssl.trustStoreType";
  private static final String SSL_TRUSTSTORE_PASSWORD = "javax.net.ssl.trustStorePassword";

  private EnhydraWrapperUtils() {}

  /**
   * Copy of Enhydra' {@link StandardDataSource} getConnection method to include, if present, the necessary javax.net.ssl
   * properties. Original code remains the same.
   */

  public static void setTLSProperties(Properties prop, TlsContextFactory tlsContextFactory) {
    if (tlsContextFactory == null)
      return;

    // Set the key store, type, and password
    if (tlsContextFactory.isKeyStoreConfigured()) {
      if (!tlsContextFactory.getKeyStoreConfiguration().getPath().isEmpty()) {
        prop.put(SSL_KEYSTORE, tlsContextFactory.getKeyStoreConfiguration().getPath());
      }

      if (!tlsContextFactory.getKeyStoreConfiguration().getType().isEmpty()) {
        prop.put(SSL_KEYSTORE_TYPE, tlsContextFactory.getKeyStoreConfiguration().getType());
      }

      if (!tlsContextFactory.getKeyStoreConfiguration().getPassword().isEmpty()) {
        prop.put(SSL_KEYSTORE_PASSWORD, tlsContextFactory.getKeyStoreConfiguration().getPassword());
      }
    }

    // Set the trust store, type, and password
    if (tlsContextFactory.isTrustStoreConfigured()) {
      if (!tlsContextFactory.getTrustStoreConfiguration().getPath().isEmpty()) {
        prop.put(SSL_TRUSTSTORE, tlsContextFactory.getTrustStoreConfiguration().getPath());
      }

      if (!tlsContextFactory.getTrustStoreConfiguration().getType().isEmpty()) {
        prop.put(SSL_TRUSTSTORE_TYPE, tlsContextFactory.getTrustStoreConfiguration().getType());
      }

      if (!tlsContextFactory.getTrustStoreConfiguration().getPassword().isEmpty()) {
        prop.put(SSL_TRUSTSTORE_PASSWORD, tlsContextFactory.getTrustStoreConfiguration().getPassword());
      }
    }
  }

}
