/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.resolver.param;

import org.mule.db.commons.api.param.ParameterType;
import org.mule.db.commons.internal.domain.connection.DbConnection;
import org.mule.db.commons.internal.domain.param.QueryParam;
import org.mule.db.commons.internal.domain.query.QueryTemplate;
import org.mule.db.commons.internal.domain.type.ArrayResolvedDbType;
import org.mule.db.commons.internal.domain.type.DbType;
import org.mule.db.commons.internal.domain.type.DbTypeManager;
import org.mule.db.commons.internal.domain.type.DynamicDbType;
import org.mule.db.commons.internal.domain.type.ResolvedDbType;
import org.mule.db.commons.internal.domain.type.StructDbType;
import org.mule.db.commons.internal.domain.type.UnknownDbType;
import org.mule.db.commons.internal.domain.type.UnknownDbTypeException;
import org.slf4j.Logger;

import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import static java.sql.Types.ARRAY;
import static java.sql.Types.STRUCT;
import static org.slf4j.LoggerFactory.getLogger;

/**
 * Resolves parameter types for standard queries
 */
public class QueryParamTypeResolver implements ParamTypeResolver {

  private static final Logger LOGGER = getLogger(QueryParamTypeResolver.class);
  private static final String UNSPECIFIED_TYPE = "UNSPECIFIED";

  private final DbTypeManager dbTypeManager;

  public QueryParamTypeResolver(DbTypeManager dbTypeManager) {
    this.dbTypeManager = dbTypeManager;
  }

  @Override
  public Map<Integer, DbType> getParameterTypes(DbConnection connection, QueryTemplate queryTemplate, List<ParameterType> types)
      throws SQLException {
    Map<Integer, DbType> paramTypes = new HashMap<>();
    PreparedStatement statement = null;
    try {
      statement = connection.getJdbcConnection().prepareStatement(queryTemplate.getSqlText());
      ParameterMetaData parameterMetaData = statement.getParameterMetaData();

      for (QueryParam queryParam : queryTemplate.getParams()) {
        int parameterTypeId = parameterMetaData.getParameterType(queryParam.getIndex());
        Optional<ParameterType> type = types.stream().filter(p -> p.getKey().equals(queryParam.getName())).findAny();
        String parameterTypeName =
            type.isPresent() ? type.get().getDbType().getName() : parameterMetaData.getParameterTypeName(queryParam.getIndex());
        DbType dbType;

        if (parameterTypeName == null || UNSPECIFIED_TYPE.equals(parameterTypeName)) {
          // Use unknown data type
          dbType = UnknownDbType.getInstance();
        } else if (type.isPresent() && !(type.get().getDbType() instanceof DynamicDbType)) {
          dbType = type.get().getDbType();
        } else {
          dbType = resolveDbType(connection, parameterTypeId, parameterTypeName);
        }

        paramTypes.put(queryParam.getIndex(), dbType);
      }
    } finally {
      if (statement != null) {
        try {
          statement.close();
        } catch (SQLException e) {
          LOGGER.warn("Could not close statement", e);
        }
      }
    }

    return paramTypes;
  }

  private DbType resolveDbType(DbConnection connection, int typeId, String typeName) {
    DbType dbType;
    try {
      dbType = dbTypeManager.lookup(connection, typeId, typeName);
      // TODO - MULE-15241 : Fix how DB Connector chooses ResolvedTypes
    } catch (UnknownDbTypeException e) {
      // Type was not found in the type manager, but the DB knows about it
      if (typeId == STRUCT) {
        // Maybe is not defined the type on the Config, but we can still use it.
        dbType = new StructDbType(typeId, typeName);
      } else if (typeId == ARRAY) {
        dbType = new ArrayResolvedDbType(typeId, typeName);
      } else {
        dbType = new ResolvedDbType(typeId, typeName);
      }
    }
    return dbType;
  }
}
