/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.consume;

import static java.lang.String.format;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.jms.commons.api.exception.JmsConsumeException;
import org.mule.jms.commons.api.exception.JmsTimeoutException;
import org.mule.jms.commons.internal.message.JmsResultFactory;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.core.api.util.func.CheckedRunnable;

import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.jms.CompletionListener;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageListener;

import org.apache.commons.lang3.time.StopWatch;
import org.slf4j.Logger;

/**
 * Wrapper implementation of a JMS {@link MessageConsumer}
 *
 * @since 1.0
 */
public final class JmsMessageConsumer implements AutoCloseable {

  private static final Double DELTA = 0.01;
  private static final Logger LOGGER = getLogger(JmsMessageConsumer.class);
  private final MessageConsumer consumer;
  private final JmsResultFactory resultFactory = JmsResultFactory.getInstance();

  public JmsMessageConsumer(MessageConsumer consumer) {
    checkArgument(consumer != null, "A non null MessageConsumer is required to use as delegate");
    this.consumer = consumer;
  }

  public void listen(MessageListener listener) throws JMSException {
    consumer.setMessageListener(listener);
  }

  public Message consume(Long maximumWaitTime) throws JMSException, JmsTimeoutException {

    if (maximumWaitTime == -1) {
      return receive();
    }

    if (maximumWaitTime == 0) {
      return receiveNoWait();
    }

    return receiveWithTimeout(maximumWaitTime);
  }

  @Override
  public void close() throws JMSException {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Closing consumer " + consumer);
    }
    consumer.close();
  }

  private Long calculateThreshold(Long timeout) {
    return timeout * DELTA < 1000 ? (long) (timeout * DELTA) : 1000;
  }

  private Message receiveWithTimeout(Long maximumWaitTime) throws JMSException, JmsTimeoutException {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug(format("Waiting for a message, timeout will be in [%s] millis", maximumWaitTime));
    }

    StopWatch timeoutValidator = new StopWatch();
    timeoutValidator.start();
    Message message = consumer.receive(maximumWaitTime);
    timeoutValidator.stop();
    // In Windows it could timeout a little earlier than it should be
    Long threshold = calculateThreshold(maximumWaitTime);

    if (message == null && timeoutValidator.getTime() >= maximumWaitTime - threshold) {
      throw new JmsTimeoutException(format("Failed to retrieve a Message. Operation timed out after %s milliseconds",
                                           maximumWaitTime));
    }
    return message;
  }

  private Message receiveNoWait() throws JMSException {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Trying to consume an immediately available message");
    }

    return consumer.receiveNoWait();
  }

  private Message receive() throws JMSException {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("No Timeout set, waiting for a message until one arrives");
    }

    return consumer.receive();
  }

  public MessageConsumer get() {
    return consumer;
  }

  /**
   * Consumes a message in a async wait. The result of the consumption will be communicated to the given {@link CompletionListener}.
   *  @param maximumWaitTime    Time to wait, in milliseconds, until a timeout is raised.
   * @param scheduler          Scheduler to create async tasks
   * @param completionListener Result callback that will be called once the message is received or a error is raised.
   */
  public void consume(Long maximumWaitTime, Scheduler scheduler, CompletionListener completionListener) {
    //TODO - The Non Blocking mechanism is behaving in a flaky way, rolling back to blocking until totally fix it
    try {
      if (maximumWaitTime == 0) {
        completionListener.onCompletion(consumer.receiveNoWait());
      } else if (maximumWaitTime == -1) {
        completionListener.onCompletion(consumer.receive());
      } else {
        completionListener.onCompletion(receiveWithTimeout(maximumWaitTime));
      }
    } catch (Exception e) {
      completionListener.onException(null, e);
    }
  }

  private void listenForMessage(Long maximumWaitTime, Scheduler scheduler, CompletionListener completionListener) {
    AtomicBoolean messageReceived = new AtomicBoolean(false);
    AtomicBoolean isClosing = new AtomicBoolean(false);
    final Reference<ScheduledFuture<?>> schedule = new Reference<>();
    try {
      consumer.setMessageListener(message -> {
        if (messageReceived.compareAndSet(true, true)) {
          return;
        }
        if (schedule.get() != null) {
          schedule.get().cancel(true);
        }
        try {
          synchronized (consumer) {
            if (!isClosing.getAndSet(true)) {
              if (consumer.getMessageListener() != null) {
                scheduler.submit((CheckedRunnable) () -> consumer.setMessageListener(null));
              }
            }
          }
        } catch (Throwable e) {
          LOGGER.warn("An unknown error occurred trying to shutdown a listener.", e);
        }
        completionListener.onCompletion(message);
      });
    } catch (JMSException e) {
      completionListener.onException(null, e);
    }

    if (maximumWaitTime > 0) {
      schedule.set(scheduler.schedule(() -> {
        if (!messageReceived.get()) {
          try {
            synchronized (consumer) {
              isClosing.getAndSet(true);
              consumer.close();
            }
            if (!messageReceived.get()) {
              completionListener.onException(null,
                                             new JmsTimeoutException(format("Failed to retrieve a Message. Operation timed out after %s milliseconds",
                                                                            maximumWaitTime)));
            }
          } catch (JMSException e) {
            completionListener.onException(null, new JmsConsumeException("Unable to listen for message.", e));
          }
        }
      }, maximumWaitTime, MILLISECONDS));
    }
  }

  private void listenForMessage(CompletionListener completionListener, Scheduler scheduler) {
    listenForMessage(-1L, scheduler, completionListener);
  }

}
