/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.source;

import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.message.Error;

import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicBoolean;

import org.slf4j.Logger;

/**
 * Default implementation of {@link JmsListenerLock} based in the usage of {@link Semaphore} to support
 * the capability of locking and unlocking the listener from different {@link Thread threads}
 *
 * @since 1.0
 * @see JmsListenerLock
 * @see JmsListener
 */
public class DefaultJmsListenerLock implements JmsListenerLock {

  private static final Logger LOGGER = getLogger(DefaultJmsListenerLock.class);

  private final AtomicBoolean isUnlocked = new AtomicBoolean(false);
  private final AtomicBoolean isFailure = new AtomicBoolean(false);
  private final Semaphore semaphore = new Semaphore(0);
  private Throwable cause;
  private Runnable action;

  /**
   * {@inheritDoc}
   */
  public void lock() {
    try {
      while (!isUnlocked.get()) {
        semaphore.acquire();

        if (action != null) {
          LOGGER.debug("About to execute action on Listener Thread");
          action.run();
          action = null;
        }
      }
      LOGGER.trace("JMS Listener lock has been unlocked.");

    } catch (InterruptedException e) {
      throw new MuleRuntimeException(createStaticMessage("The JMS Listener Lock has been interrupted."), cause);
    }

    synchronized (isUnlocked) {
      if (isFailure.get()) {
        throw new MuleRuntimeException(cause);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void unlockWithFailure(Error error) {
    synchronized (isUnlocked) {
      isFailure.compareAndSet(false, true);
      isUnlocked.compareAndSet(false, true);
      if (error != null) {
        cause = error.getCause();
      } else {
        //This should not happen, but sometimes happen
        cause = new MuleRuntimeException(createStaticMessage("Unknown error."));
      }
    }
    releaseIfNecessary();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void unlockWithFailure() {
    synchronized (isUnlocked) {
      isFailure.compareAndSet(false, true);
      isUnlocked.compareAndSet(false, true);
      cause =
          new MuleRuntimeException(createStaticMessage("The JMS Lock has been unlocked indicating that a session recover should be executed."));
    }
    releaseIfNecessary();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void unlock() {
    synchronized (isUnlocked) {
      isFailure.compareAndSet(true, false);
      isUnlocked.compareAndSet(false, true);
    }
    releaseIfNecessary();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void executeOnListenerThread(Runnable runnable) {
    LOGGER.trace("About to execute runnable on listener thread.");
    synchronized (isUnlocked) {
      if (isUnlocked.get()) {
        LOGGER.trace("Listener thread is not more available, running on current thread.");
        runnable.run();
      } else {
        this.action = runnable;
        releaseIfNecessary();
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void init() {
    isUnlocked.set(false);
    isFailure.set(false);
    cause = null;
    action = null;
  }

  private void releaseIfNecessary() {
    synchronized (semaphore) {
      if (isSemaphoreWithOutAvailablePermits()) {
        semaphore.release();
      }
    }
  }

  private boolean isSemaphoreWithOutAvailablePermits() {
    return semaphore.availablePermits() == 0;
  }

}
