/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.source;

import org.mule.jms.commons.api.RequestReplyPattern;
import org.mule.jms.commons.api.destination.JmsDestination;
import org.mule.jms.commons.api.message.JmsMessageBuilder;
import org.mule.jms.commons.internal.publish.PublisherParameters;
import org.mule.runtime.extension.api.runtime.parameter.OutboundCorrelationStrategy;

import java.util.concurrent.TimeUnit;

import javax.jms.Message;

/**
 * Enables the creation of an outgoing {@link Message} along with the response configuration.
 * Users must use this builder to create a message response instance.
 *
 * @since 1.0.0
 */
public interface JmsResponseMessageBuilder<D extends JmsDestination> extends JmsMessageBuilder<D>, PublisherParameters {

  /**
   * Whether or not the delivery should be done with a persistent configuration
   */
  boolean isPersistentDelivery();

  /**
   * The default JMSPriority value to be used when sending the message
   */
  Integer getPriority();

  /**
   * Defines the default time the message will be in the broker before it expires and is discarded
   */
  Long getTimeToLive();

  /**
   * Time unit to be used in the timeToLive configurations
   */
  TimeUnit getTimeToLiveUnit();

  /**
   * If true; the Message will be flagged to avoid generating its MessageID
   */
  boolean isDisableMessageId();

  /**
   * If true; the Message will be flagged to avoid generating its sent Timestamp
   */
  boolean isDisableMessageTimestamp();

  /**
   * Only used by JMS 2.0. Sets the delivery delay to be applied in order to postpone the Message delivery
   */
  Long getDeliveryDelay();

  /**
   * Time unit to be used in the deliveryDelay configurations
   */
  TimeUnit getDeliveryDelayUnit();

  /**
   * Options on whether to include an outbound correlation id or not
   *
   * @since 1.1.0
   */
  OutboundCorrelationStrategy getSendCorrelationId();

  /**
   * Returns the Request Reply Pattern to use
   *
   * @since 1.6.0
   */
  RequestReplyPattern getRequestReplyPattern();

  /**
   * If true no automatic response will be sent in case of Jms replyTo header presence.
   *
   * @since 1.9.0
   * @return status of the Jms replyTo ignore flag.
   */
  default boolean isReplyToIgnored() {
    return false;
  }

}
