/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.connection.provider;

import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.mule.jms.commons.api.connection.JmsSpecification.JMS_1_0_2b;
import static org.mule.jms.commons.api.connection.JmsSpecification.JMS_2_0;
import static org.mule.jms.commons.internal.common.JmsCommons.closeQuietly;
import static org.mule.jms.commons.internal.common.JmsCommons.stopQuietly;
import static org.mule.runtime.api.connection.ConnectionValidationResult.failure;
import static org.mule.runtime.api.connection.ConnectionValidationResult.success;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.disposeIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.stopIfNeeded;
import static org.slf4j.LoggerFactory.getLogger;

import org.apache.commons.lang3.concurrent.BasicThreadFactory;
import org.mule.jms.commons.api.connection.DefaultReconnectionManagerProvider;
import org.mule.jms.commons.api.connection.JmsReconnectionManager;
import org.mule.jms.commons.api.connection.JmsSpecification;
import org.mule.jms.commons.api.connection.caching.CachingStrategy;
import org.mule.jms.commons.internal.connection.IBMJmsCachingConnectionFactory;
import org.mule.jms.commons.internal.connection.JmsCachingConnectionFactory;
import org.mule.jms.commons.internal.connection.JmsConnection;
import org.mule.jms.commons.internal.connection.JmsTransactionalConnection;
import org.mule.jms.commons.internal.connection.XaJmsTransactionalConnection;
import org.mule.jms.commons.internal.connection.exception.CompositeJmsExceptionListener;
import org.mule.jms.commons.internal.connection.param.GenericConnectionParameters;
import org.mule.jms.commons.internal.connection.param.XaPoolParameters;
import org.mule.jms.commons.internal.connection.session.JmsSessionManager;
import org.mule.jms.commons.internal.support.JmsSupport;
import org.mule.jms.commons.internal.support.JmsSupportFactory;
import org.mule.runtime.api.connection.CachedConnectionProvider;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.connection.PoolingConnectionProvider;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;

import java.util.concurrent.ExecutorService;

import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;

import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.JMSException;

import org.slf4j.Logger;


/**
 * Base implementation of a {@link PoolingConnectionProvider} for {@link JmsConnection}s
 *
 * @since 1.0
 */
public class JmsConnectionProvider implements CachedConnectionProvider<JmsTransactionalConnection>, Initialisable, Disposable {

  private static final int VALIDATION_TIMEOUT = 10;
  private static final String VALIDATION_THREAD_NAME_PATTERN = "connectionValidationThread-%d";
  private static final Logger LOGGER = getLogger(JmsConnectionProvider.class);
  private final JmsSessionManager jmsSessionManager;
  private final JmsSpecification specification;
  private final GenericConnectionParameters connectionParameters;
  private final XaPoolParameters xaPoolParameters;
  private final CachingStrategy cachingStrategy;
  private final boolean isIBM;

  private XaJmsTransactionalConnection xaJmsTransactionalConnection;
  private final ConnectionFactoryDecorator factoryDecorator;
  private final Supplier<ConnectionFactory> connectionFactorySupplier;
  private final boolean isXa;
  private final JmsSupportFactory jmsSupportFactory;
  private final String configName;
  private JmsReconnectionManager reconnectionManager;

  private CompositeJmsExceptionListener exceptionListener = new CompositeJmsExceptionListener();

  /**
   * Used to ignore handling of ExceptionListener#onException when in the process of disconnecting
   */
  private final AtomicBoolean disconnecting = new AtomicBoolean(false);

  private JmsSupport jmsSupport;
  private ConnectionFactory jmsConnectionFactory;
  private boolean isCacheEnabled = false;

  public JmsConnectionProvider(JmsSessionManager jmsSessionManager,
                               Supplier<ConnectionFactory> connectionFactorySupplier,
                               JmsSpecification specification,
                               GenericConnectionParameters connectionParameters,
                               XaPoolParameters xaPoolParameters,
                               CachingStrategy cachingStrategy,
                               boolean isXa,
                               JmsSupportFactory jmsSupportFactory,
                               ConnectionFactoryDecoratorFactory factoryDecoratorFactory,
                               String configName,
                               boolean isIBM,
                               JmsReconnectionManager reconnectionManager)
      throws InitialisationException {
    this.jmsSessionManager = jmsSessionManager;
    this.connectionFactorySupplier = connectionFactorySupplier;
    this.specification = specification;
    this.connectionParameters = connectionParameters;
    this.xaPoolParameters = xaPoolParameters;
    this.cachingStrategy = cachingStrategy;
    this.isXa = isXa;
    this.jmsSupportFactory = jmsSupportFactory;
    this.factoryDecorator = factoryDecoratorFactory.create();
    this.configName = configName;
    this.isIBM = isIBM;
    this.reconnectionManager = reconnectionManager;

    initialise();
  }

  public JmsConnectionProvider(JmsSessionManager jmsSessionManager,
                               Supplier<ConnectionFactory> connectionFactorySupplier,
                               JmsSpecification specification,
                               GenericConnectionParameters connectionParameters,
                               XaPoolParameters xaPoolParameters,
                               CachingStrategy cachingStrategy,
                               boolean isXa,
                               JmsSupportFactory jmsSupportFactory,
                               ConnectionFactoryDecoratorFactory factoryDecoratorFactory,
                               String configName,
                               boolean isIBM)
      throws InitialisationException {
    this(jmsSessionManager,
         connectionFactorySupplier,
         specification,
         connectionParameters,
         xaPoolParameters,
         cachingStrategy,
         isXa,
         jmsSupportFactory,
         factoryDecoratorFactory,
         configName,
         isIBM,
         new DefaultReconnectionManagerProvider());
  }


  public JmsConnectionProvider(JmsSessionManager jmsSessionManager,
                               Supplier<ConnectionFactory> connectionFactorySupplier,
                               JmsSpecification specification,
                               GenericConnectionParameters connectionParameters,
                               XaPoolParameters xaPoolParameters,
                               CachingStrategy cachingStrategy,
                               boolean isXa,
                               JmsSupportFactory jmsSupportFactory,
                               ConnectionFactoryDecoratorFactory factoryDecoratorFactory,
                               String configName)
      throws InitialisationException {
    this(jmsSessionManager, connectionFactorySupplier, specification, connectionParameters, xaPoolParameters,
         cachingStrategy, isXa, jmsSupportFactory, factoryDecoratorFactory, configName, false);
  }

  /**
   * Template method for obtaining the {@link ConnectionFactory} to be used for creating the {@link JmsConnection}s
   * 
   * @return an instance of {@link ConnectionFactory} to be used for creating the {@link JmsConnection}s
   * @throws Exception if an error occurs while creting the {@link ConnectionFactory}
   */
  public ConnectionFactory getConnectionFactory() throws Exception {
    ConnectionFactory connectionFactory = connectionFactorySupplier.get();
    return factoryDecorator.decorate(connectionFactory, configName, true,
                                     connectionParameters,
                                     xaPoolParameters,
                                     exceptionListener);
  }



  @Override
  public void initialise() throws InitialisationException {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug(format("Executing initialise for [%s]", getClass().getName()));
    }
    try {
      createJmsSupport();
      initialiseConnectionFactory();

    } catch (Exception e) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug(format("Failed to initialise [%s]: ", getClass().getName()), e);
      }
      throw new InitialisationException(e, this);
    }
  }

  @Override
  public JmsTransactionalConnection connect() throws ConnectionException {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Connection Started");
    }

    disconnecting.set(false);

    try {
      Connection connection = createConnection();
      connection.start();
      if (isXa) {
        xaJmsTransactionalConnection =
            new XaJmsTransactionalConnection(jmsSupport, connection, jmsSessionManager, exceptionListener);
        return xaJmsTransactionalConnection;
      } else {
        return new JmsTransactionalConnection(jmsSupport, connection, jmsSessionManager, exceptionListener);
      }
    } catch (Exception e) {
      try {
        // If connection throws an exception on start and connection is cached in ConnectionFactory then
        // stop/reset connection now.
        stopIfNeeded(jmsConnectionFactory);

      } catch (MuleException factoryStopException) {
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Failed to reset cached connection: ", factoryStopException);
        }
      }

      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Failed create connection: ", e);
      }

      throw new ConnectionException(e);
    }
  }

  @Override
  public ConnectionValidationResult validate(JmsTransactionalConnection jmsConnection) {
    LOGGER.debug("Validating connection");
    try {
      // According to javax.jms.Connection#start javadoc:
      // 'a call to start on a connection that has already been started is ignored'
      // and exception is thrown 'if the JMS provider fails to start'
      // thus, if the connection is valid, we should be able to re-start it even if the 'connect'
      // method did it already.
      // Actually the previous comment is not enough since BTI will IBM MQ does not fails on a start() over a connection that may
      // be invalid.
      // Still the start() needs to get called in case there object Connection from JMS actually failed to connect and has to
      // reconnect
      Connection connection = jmsConnection.get();
      connection.start();

      ConnectionValidationResult res = validateWithTimeout(jmsConnection);
      if (res.isValid()) {
        return res;
      }
      return failure("Connection validation failure", res.getException());
    } catch (Exception e1) {
      LOGGER.error(e1.getMessage(), e1);
      if (jmsConnection instanceof XaJmsTransactionalConnection) {
        LOGGER.debug("Closing XA transaction");
        closeQuietly(jmsConnection.get());
        try {
          disposeIfNeeded(jmsConnectionFactory, LOGGER);
          initialiseConnectionFactory();
          xaJmsTransactionalConnection.close();
          LOGGER.debug("Create Connection");
          Connection connection = createConnection();
          connection.start();
          xaJmsTransactionalConnection.setConnection(connection);

          ConnectionValidationResult res2 = validateWithTimeout(xaJmsTransactionalConnection);
          if (res2.isValid()) {
            LOGGER.debug("Validation succeeded");
            return success();
          }
          LOGGER.debug("Connection Validation Failure");
          return failure("Connection Validation Failure", res2.getException());

        } catch (Exception e2) {
          // Do nothing, just return failure with original error
          LOGGER.debug("Handled exception on connection validation", e2);
          // TODO - MULE-11433: Useful error never informed by connectivity testing
          return failure("Invalid connection provided: Connection could not be started.", e2);
        }
      }
      LOGGER.debug("Could not validate connection - No XA Transactiongit  -", e1);
      return failure("Could not validate connection", e1);
    }
  }


  private ConnectionValidationResult validateWithTimeout(JmsTransactionalConnection jmsConnection)
      throws InterruptedException, ExecutionException, TimeoutException {
    LOGGER.debug("validateWithTimeout");
    BasicThreadFactory threadFactory = new BasicThreadFactory.Builder().namingPattern(VALIDATION_THREAD_NAME_PATTERN).build();
    ExecutorService executor = Executors.newSingleThreadExecutor(threadFactory);
    Future<ConnectionValidationResult> future = null;
    try {
      LOGGER.debug("validateWithTimeout - Submit Validation Task");
      future = executor.submit(new JmsConnectionValidationTask(jmsConnection));
      ConnectionValidationResult res = future.get(VALIDATION_TIMEOUT, TimeUnit.SECONDS);
      LOGGER.debug("validateWithTimeout - Receive Validation Task");
      return res;
    } catch (TimeoutException e) {
      LOGGER.debug("JmsConnectionProvider.validateWithTimeout", e);
      future.cancel(true);
      return failure(e.getMessage(), e);
    } finally {
      LOGGER.debug("validateWithTimeout - finally");
      executor.shutdownNow();
    }
  }

  @Override
  public void disconnect(JmsTransactionalConnection jmsConnection) {
    LOGGER.debug("Disconnection Started");

    // This invocations will be ignored when using a CachingConnectionFactory,
    // since a single connection is cached
    disconnecting.set(true);
    exceptionListener = new CompositeJmsExceptionListener();
    doStop(jmsConnection);
    doClose(jmsConnection);
  }

  protected void doStop(JmsConnection jmsConnection) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug(format("Perform doStop: [%s]", getClass().getName()));
    }

    stopQuietly(jmsConnection);
    disposeIfNeeded(jmsConnection, LOGGER);
    stopQuietly(jmsConnectionFactory);
    disposeIfNeeded(jmsConnectionFactory, LOGGER);
  }

  protected void doClose(JmsConnection jmsConnection) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug(format("Perform doClose: [%s]", getClass().getName()));
    }
    disposeIfNeeded(jmsConnection, LOGGER);
  }

  @Override
  public void dispose() {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug(format("Disposing [%s]", getClass().getName()));
    }
    disposeIfNeeded(jmsConnectionFactory, LOGGER);
  }

  private void initialiseConnectionFactory() throws Exception {

    LOGGER.debug("Initialising Connection Factory");

    ConnectionFactory targetFactory = getConnectionFactory();

    initialiseIfNeeded(targetFactory);

    if (cachingStrategy.appliesTo(targetFactory) && cachingStrategy.strategyConfiguration().isPresent()) {

      if (isXa) {
        throw new IllegalStateException("An XA ConnectionFactory cannot be used with a caching connection.");
      }

      isCacheEnabled = true;

      String username = getConnectionParameters().getUsername();
      String password = getConnectionParameters().getPassword();
      String clientId = getConnectionParameters().getClientId();

      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug(format("Using CachingConnectionFactory wrapper with: username:[%s], password:[%s], clientId:[%s]",
                            username, password, clientId));
      }
      if (isIBM) {

        jmsConnectionFactory =
            new IBMJmsCachingConnectionFactory(targetFactory, username, password, clientId,
                                               cachingStrategy.strategyConfiguration().get(),
                                               jmsSupport, exceptionListener, reconnectionManager);
      } else {
        jmsConnectionFactory =
            new JmsCachingConnectionFactory(targetFactory, username, password, clientId,
                                            cachingStrategy.strategyConfiguration().get(),
                                            jmsSupport, exceptionListener);
      }

      initialiseIfNeeded(jmsConnectionFactory);
    } else {
      LOGGER.debug("Skip CachingConnectionFactory Wrapper");

      jmsConnectionFactory = targetFactory;
    }
  }

  /**
   * A jmsConnectionFactory method to create various JmsSupport class versions.
   *
   * @see JmsSupport
   */
  protected void createJmsSupport() {
    JmsSpecification specification = getSpecification();
    if (JMS_1_0_2b.equals(specification)) {
      jmsSupport = jmsSupportFactory.create102bSupport();

    } else if (JMS_2_0.equals(specification)) {
      jmsSupport = jmsSupportFactory.create20Support();

    } else {
      jmsSupport = jmsSupportFactory.create11Support();
    }

    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug(format("JMS Support set to [%s]", jmsSupport.getSpecification().getName()));
    }
  }

  private Connection createConnection() throws JMSException {

    String username = getConnectionParameters().getUsername();
    String password = getConnectionParameters().getPassword();

    Connection connection;

    if (isCacheEnabled || isBlank(username)) {
      connection = jmsSupport.createConnection(jmsConnectionFactory);
    } else {
      connection = jmsSupport.createConnection(jmsConnectionFactory, username, password);
    }

    if (connection == null) {
      throw new IllegalStateException("An error occurred, Connection cannot be null after creation");
    }

    if (!isCacheEnabled) {
      String clientId = getConnectionParameters().getClientId();
      if (!isBlank(clientId) && !clientId.equals(connection.getClientID())) {
        connection.setClientID(clientId);
      }

      if (connection.getExceptionListener() == null) {
        try {
          connection.setExceptionListener(exceptionListener);
        } catch (Exception e) {
          if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("An error occurred while setting the ExceptionListener. "
                + "No ExceptionListener is available in a Java EE web or EJB application. ", e);
          }
        }
      }
    }
    return connection;
  }

  public GenericConnectionParameters getConnectionParameters() {
    return connectionParameters;
  }

  public JmsSupport getJmsSupport() {
    return jmsSupport;
  }

  protected void setJmsSupport(JmsSupport jmsSupport) {
    this.jmsSupport = jmsSupport;
  }

  public JmsSpecification getSpecification() {
    return specification;
  }

}
