/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.source.push;

import static java.lang.String.format;
import static org.mule.jms.commons.internal.config.InternalAckMode.TRANSACTED;
import static org.mule.jms.commons.internal.source.push.JmsMessageListenerDelegate.JMS_LOCK_VAR;
import static org.slf4j.LoggerFactory.getLogger;
import org.mule.jms.commons.api.connection.JmsSpecification;
import org.mule.jms.commons.api.message.DefaultJmsAttributes;
import org.mule.jms.commons.internal.config.InternalAckMode;
import org.mule.jms.commons.internal.config.JmsConfig;
import org.mule.jms.commons.internal.connection.JmsConnection;
import org.mule.jms.commons.internal.connection.JmsTransactionalConnection;
import org.mule.jms.commons.internal.connection.session.JmsSession;
import org.mule.jms.commons.internal.connection.session.JmsSessionManager;
import org.mule.jms.commons.internal.source.JmsConnectionExceptionResolver;
import org.mule.jms.commons.internal.source.JmsListener;
import org.mule.jms.commons.internal.source.JmsListenerLock;
import org.mule.jms.commons.internal.source.JmsMessageDispatcher;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionProvider;
import org.mule.runtime.api.tx.TransactionException;
import org.mule.runtime.core.api.transaction.TransactionCoordination;
import org.mule.runtime.extension.api.runtime.source.SourceCallback;
import org.mule.runtime.extension.api.runtime.source.SourceCallbackContext;

import java.util.concurrent.atomic.AtomicBoolean;

import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;

import org.slf4j.Logger;

/**
 * {@link MessageListener} for the {@link JmsListener} to subscribe to a TOPIC or QUEUE receives {@link Message} and dispatch
 * those messages through the flow.
 *
 * @since 1.0
 */
public final class JmsMessageListener<T extends DefaultJmsAttributes> implements MessageListener {

  private static final Logger LOGGER = getLogger(JmsMessageListener.class);

  private final JmsSession session;
  private final SourceCallback<Object, T> sourceCallback;
  private final JmsListenerLock jmsLock;
  private final JmsMessageDispatcher messageDispatcher;
  private final JmsSessionManager sessionManager;
  private final JmsConnectionExceptionResolver exceptionResolver;
  private final boolean isTransactedSession;
  private ConnectionProvider<JmsTransactionalConnection> connectionProvider;
  private AtomicBoolean enabled = new AtomicBoolean(true);

  /**
   * Creates a new instance of a {@link JmsMessageListener}
   *
   * @param session the session to create the JMS Consumer
   * @param config JMS
   * @param jmsLock the lock to use to synchronize the message dispatch
   * @param sessionManager manager to store the session and ACK ID of each dispatched message
   * @param sourceCallback callback use to dispatch the {@link Message} to the mule flow
   * @param specification JMS Support that communicates the used specification
   * @param ackMode Acknowledgement mode to use to consume the messages
   * @param encoding Default encoding if the consumed message doesn't provide one
   * @param contentType Default contentType if the consumed message doesn't provide one
   * @param connectionProvider JMS Connection provider which will provide {@link JmsConnection} to bind into the
   *        {@link SourceCallbackContext}
   * @param exceptionResolver Exception resolver used in case some broker specific conditions qualify as a connection issue
   */
  JmsMessageListener(JmsSession session,
                     JmsConfig config,
                     JmsListenerLock jmsLock,
                     JmsSessionManager sessionManager,
                     SourceCallback<Object, T> sourceCallback,
                     JmsSpecification specification,
                     InternalAckMode ackMode,
                     String encoding,
                     String contentType,
                     ConnectionProvider<JmsTransactionalConnection> connectionProvider,
                     JmsConnectionExceptionResolver exceptionResolver) {
    this.session = session;
    this.sourceCallback = sourceCallback;
    this.jmsLock = jmsLock;
    this.sessionManager = sessionManager;
    this.connectionProvider = connectionProvider;
    this.exceptionResolver = exceptionResolver;
    this.isTransactedSession = ackMode.equals(TRANSACTED);
    this.messageDispatcher = new JmsMessageDispatcher(config, contentType, encoding, specification, () -> session, ackMode,
                                                      sessionManager, sourceCallback, jmsLock, exceptionResolver);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onMessage(Message message) {
    if (enabled.get()) {
      doOnMessage(message);
    } else {
      LOGGER.warn(getListenerIsBeingClosedMessage(message));
      try {
        if (isTransactedSession) {
          // No TX has been created yet on Mule Runtime, but the Session already started the TX, so just closing rollback it.
          session.get().rollback();
        } else {
          tryRecoverSession();
        }
      } catch (JMSException e) {
        LOGGER.error("An error occurred trying to rollback the current session {}", session);
      }
    }
  }

  protected void doOnMessage(Message message) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Received message on session [{}]", session.get().toString());
      logNewMessageIds(message);
    }

    SourceCallbackContext context = sourceCallback.createContext();

    try {
      context.bindConnection(connectionProvider.connect());
    } catch (ConnectionException | TransactionException e) {
      JmsListener.notifyIfConnectionProblem(sourceCallback, e, exceptionResolver);
    }

    if (isTransactedSession) {
      sessionManager.bindToTransaction(session);
    }

    jmsLock.init();
    context.addVariable(JMS_LOCK_VAR, jmsLock);

    try {
      messageDispatcher.dispatchMessage(message, context);
      waitForMessageToBeProcessed(jmsLock);
    } catch (Exception e) {
      LOGGER.warn("An error occurred processing the message, returning it to the queue", e);
      if (isTransactedSession) {
        TransactionCoordination.getInstance().rollbackCurrentTransaction();
      } else {
        tryRecoverSession();
      }
    }
  }

  protected void tryRecoverSession() {
    try {
      session.get().recover();
    } catch (JMSException jmsException) {
      // As JMS Spec says onMessage() method should never throw exceptions, should handle problems internally.
      LOGGER.warn("An error occurred trying to recover the session because of a failure message processing.", jmsException);
    }
  }

  protected void logNewMessageIds(Message message) {
    try {
      LOGGER.debug("New Message: MessageID [{}]. CorrelationID [{}]", message.getJMSMessageID(),
                   message.getJMSCorrelationID());
    } catch (JMSException e) {
      LOGGER.debug("New Message: Message/Correlation ID's could not be extracted because of: {}", e);
    }
  }

  private void waitForMessageToBeProcessed(JmsListenerLock jmsLock) {
    LOGGER.debug("Waiting for message to be processed through flow");
    jmsLock.lock();
    LOGGER.debug("Resuming message consuming");
  }

  public void setEnabled(boolean state) {
    LOGGER.debug("Switching message listener [{}] enabled flag to [{}]", this, state);
    enabled.set(state);
  }

  private String getListenerIsBeingClosedMessage(Message message) {
    try {
      String jmsMessageID = message.getJMSMessageID();
      String jmsCorrelationID = message.getJMSCorrelationID();
      return format("Message listener is being closed, the current message with ID %S and Correlation ID %S " +
          "is not going to be processed and is returned to the Queue.", jmsMessageID, jmsCorrelationID);
    } catch (Exception e) {
      return "Message listener is being closed, the current message is not going to be processed and is returned to the Queue.";
    }
  }

  public boolean isEnabled() {
    return enabled.get();
  }
}
