/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.operation;

import static java.lang.String.format;
import static org.mule.jms.commons.internal.common.JmsCommons.closeQuietly;
import static org.mule.jms.commons.internal.common.JmsCommons.evaluateMessageAck;
import static org.mule.jms.commons.internal.common.JmsCommons.getDestinationType;
import static org.mule.jms.commons.internal.common.JmsCommons.resolveMessageContentType;
import static org.mule.jms.commons.internal.common.JmsCommons.resolveMessageEncoding;
import static org.mule.jms.commons.internal.common.JmsCommons.resolveOverride;
import static org.mule.jms.commons.internal.common.JmsCommons.toInternalAckMode;
import static org.mule.jms.commons.internal.config.InternalAckMode.AUTO;
import static org.slf4j.LoggerFactory.getLogger;

import javax.inject.Inject;
import javax.jms.CompletionListener;
import javax.jms.Destination;
import javax.jms.IllegalStateException;
import javax.jms.JMSException;
import javax.jms.JMSSecurityException;
import javax.jms.Message;
import javax.jms.Queue;
import javax.jms.TemporaryQueue;
import javax.jms.TemporaryTopic;
import javax.jms.Topic;

import org.mule.jms.commons.api.RequestReplyPattern;
import org.mule.jms.commons.api.RequestReplyPatternWrapper;
import org.mule.jms.commons.api.config.DefaultJmsProducerConfig;
import org.mule.jms.commons.api.destination.ConsumerType;
import org.mule.jms.commons.api.destination.QueueConsumer;
import org.mule.jms.commons.api.destination.TopicConsumer;
import org.mule.jms.commons.api.exception.JmsConsumeException;
import org.mule.jms.commons.api.exception.JmsExtensionException;
import org.mule.jms.commons.api.exception.JmsPublishConsumeErrorTypeProvider;
import org.mule.jms.commons.api.exception.JmsPublishException;
import org.mule.jms.commons.api.exception.JmsSecurityException;
import org.mule.jms.commons.api.message.DefaultJmsMessageBuilder;
import org.mule.jms.commons.api.message.JmsAttributes;
import org.mule.jms.commons.api.message.JmsMessageBuilder;
import org.mule.jms.commons.internal.config.InternalAckMode;
import org.mule.jms.commons.internal.config.JmsConfig;
import org.mule.jms.commons.internal.connection.JmsConnection;
import org.mule.jms.commons.internal.connection.session.JmsSession;
import org.mule.jms.commons.internal.connection.session.JmsSessionManager;
import org.mule.jms.commons.internal.consume.JmsConsumeParameters;
import org.mule.jms.commons.internal.consume.JmsMessageConsumer;
import org.mule.jms.commons.internal.message.JmsResultFactory;
import org.mule.jms.commons.internal.metadata.JmsOutputResolver;
import org.mule.jms.commons.internal.operation.publishconsume.ProducerResourceCloserAction;
import org.mule.jms.commons.internal.publish.JmsMessageProducer;
import org.mule.jms.commons.internal.publish.JmsPublishParameters;
import org.mule.jms.commons.internal.support.JmsSupport;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.core.api.util.func.CheckedFunction;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.metadata.OutputResolver;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.ConfigOverride;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.annotation.param.ParameterGroup;
import org.mule.runtime.extension.api.annotation.param.display.Placement;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.parameter.CorrelationInfo;
import org.mule.runtime.extension.api.runtime.parameter.OutboundCorrelationStrategy;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;
import org.slf4j.Logger;

/**
 * Operation that allows the user to send a message to a JMS {@link Destination} and waits for a response either to the provided
 * {@code ReplyTo} destination or to a temporary {@link Destination} created dynamically
 *
 * @since 1.0
 */
public class JmsPublishConsume implements Disposable {

  private static final Logger LOGGER = getLogger(JmsPublishConsume.class);
  private final Scheduler scheduler;
  private JmsResultFactory resultFactory = JmsResultFactory.getInstance();

  @Inject
  private JmsSessionManager sessionManager;

  private RequestReplyPatternWrapper requestReplyPatternWrapper = new DefaultRequestReplyPatternWrapper();

  public JmsPublishConsume(JmsSessionManager sessionManager, SchedulerService schedulerService) {
    this.sessionManager = sessionManager;
    this.scheduler = schedulerService.ioScheduler();
  }

  public JmsPublishConsume(JmsSessionManager sessionManager, SchedulerService schedulerService,
                           RequestReplyPatternWrapper customRequestReplyPattern) {
    this.sessionManager = sessionManager;
    this.scheduler = schedulerService.ioScheduler();
    this.requestReplyPatternWrapper = customRequestReplyPattern;
  }

  /**
   * Operation that allows the user to send a message to a JMS {@link Destination} and waits for a response either to the provided
   * {@code ReplyTo} destination or to a temporary {@link Destination} created dynamically
   *
   * @param config            the current {@link DefaultJmsProducerConfig}
   * @param connection        the current {@link JmsConnection}
   * @param destination       the name of the {@link Destination} where the {@link Message} should be sent
   * @param messageBuilder    the {@link DefaultJmsMessageBuilder} used to create the {@link Message} to be sent
   * @param publishParameters Parameter group that lets override the publish configuration
   * @param consumeParameters Parameter group that lets override the consume configuration
   * @param sendCorrelationId options on whether to include an outbound correlation id or not
   * @param correlationInfo   the current message's correlation info
   * @return a {@link Result} with the reply {@link Message} content as {@link Result#getOutput} and its properties and headers as
   *         {@link Result#getAttributes}
   * @throws JmsExtensionException if an error occurs
   */
  @OutputResolver(output = JmsOutputResolver.class)
  @Throws(JmsPublishConsumeErrorTypeProvider.class)
  public void publishConsume(@Config JmsConfig config,
                             @Connection JmsConnection connection,
                             @Placement(
                                 order = 0) @Summary("The name of the Queue destination where the Message should be sent") String destination,
                             @Placement(
                                 order = 1) @Summary("A builder for the message that will be published") @ParameterGroup(
                                     name = "Message",
                                     showInDsl = true) JmsMessageBuilder messageBuilder,
                             @Placement(order = 2) @ParameterGroup(
                                 name = "Publish Configuration",
                                 showInDsl = true) JmsPublishParameters publishParameters,
                             @Placement(order = 3) @ParameterGroup(
                                 name = "Consume Configuration",
                                 showInDsl = true) JmsConsumeParameters consumeParameters,
                             @ConfigOverride OutboundCorrelationStrategy sendCorrelationId,
                             CorrelationInfo correlationInfo,
                             RequestReplyPattern requestReplyPattern,
                             CompletionCallback<Object, JmsAttributes> completionCallback)
      throws JmsExtensionException {

    JmsSession producerSession = null;
    Message message;
    ConsumerType replyConsumerType;
    Destination replyDestination;
    InternalAckMode resolvedAckMode = resolveOverride(toInternalAckMode(config.getConsumerConfig().getAckMode()),
                                                      toInternalAckMode(consumeParameters.getAckMode()));

    JmsMessageProducer producer;
    try {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Begin [publish] of [publishConsume] to the QUEUE: [" + destination + "]");
      }

      JmsSupport jmsSupport = connection.getJmsSupport();
      producerSession = connection.createSession(resolvedAckMode, false);

      message = messageBuilder.build(jmsSupport, sendCorrelationId, correlationInfo, producerSession.get(), config);

      replyDestination = setReplyDestination(messageBuilder, producerSession, jmsSupport, message);
      replyConsumerType = getConsumerType(replyDestination);

      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Message built, sending message to the QUEUE:  [" + destination + "]");
      }

      Destination publishDestination = jmsSupport.createDestination(producerSession.get(), destination, false, config);
      producer = connection.createProducer(producerSession, publishDestination, false);

      AsyncPublishCompletionListener completionListener =
          new AsyncPublishCompletionListener(publishDestination, replyConsumerType, replyDestination, consumeParameters,
                                             resolvedAckMode.equals(AUTO) ? InternalAckMode.IMMEDIATE : resolvedAckMode,
                                             completionCallback, requestReplyPattern, connection, config,
                                             requestReplyPatternWrapper,
                                             new ProducerResourceCloserAction(producerSession, producer,
                                                                              config.getResourceReleaserScheduler()));
      producer.publish(message, publishParameters,
                       completionListener);

    } catch (JMSSecurityException e) {
      String msg =
          format("A security error occurred while sending a message to the QUEUE: [%s]: %s", destination, e.getMessage());
      completionCallback.error(new JmsSecurityException(msg, e));
    } catch (IllegalStateException e) {
      boolean isConnectionError = producerSession == null;
      String msg = format((isConnectionError ? "A connection error" : "An error")
          + " occurred while sending a message to the QUEUE: [%s]: %s", destination, e.getMessage());
      completionCallback.error(isConnectionError ? new ConnectionException(e) : new JmsPublishException(msg, e));
    } catch (Exception e) {
      String msg = format("An error occurred while sending a message to the QUEUE: [%s]: %s", destination, e.getMessage());
      completionCallback.error(new JmsPublishException(msg, e));
    }
  }

  /**
   * {@link CompletionListener} for the Publish step of the Publish-Consume Operation. This will receive asynchronously the
   * confirmation of the sent message and will start to wait for a reply.
   */
  private class AsyncPublishCompletionListener implements CompletionListener {

    private final Destination publishDestination;
    private final ConsumerType replyConsumerType;
    private final Destination replyDestination;
    private final CheckedFunction<String, JmsMessageConsumer> consumerFactory;
    private JmsMessageConsumer consumer;
    private final JmsConsumeParameters consumeParameters;
    private final InternalAckMode resolvedAckMode;
    private final JmsSession consumerSession;
    private final CompletionCallback<Object, JmsAttributes> completionCallback;
    private RequestReplyPattern requestReplyPattern;
    private final JmsConnection connection;
    private final JmsConfig config;
    private String replyDestinationName;
    private RequestReplyPatternWrapper requestReplyWrapper = new DefaultRequestReplyPatternWrapper();
    private ProducerResourceCloserAction resourceCloserAction;

    AsyncPublishCompletionListener(Destination publishDestination,
                                   ConsumerType replyConsumerType,
                                   Destination replyDestination,
                                   JmsConsumeParameters consumeParameters,
                                   InternalAckMode resolvedAckMode,
                                   CompletionCallback<Object, JmsAttributes> completionCallback,
                                   RequestReplyPattern requestReplyPattern,
                                   JmsConnection connection,
                                   JmsConfig config,
                                   RequestReplyPatternWrapper requestReplyWrapper,
                                   ProducerResourceCloserAction resourceCloserAction)
        throws JMSException {
      this.publishDestination = publishDestination;
      this.replyConsumerType = replyConsumerType;
      this.replyDestination = replyDestination;
      this.consumeParameters = consumeParameters;
      this.resolvedAckMode = resolvedAckMode;
      this.completionCallback = completionCallback;
      this.requestReplyPattern = requestReplyPattern;
      this.connection = connection;
      this.config = config;
      this.requestReplyWrapper = requestReplyWrapper;
      this.replyDestinationName = getReplyDestinationName(replyDestination, replyConsumerType);
      this.consumerSession = connection.createSession(resolvedAckMode, false);
      this.consumerFactory =
          selector -> connection.createConsumer(consumerSession, replyDestination, selector, replyConsumerType, true);
      this.resourceCloserAction = resourceCloserAction;
    }

    @Override
    public void onCompletion(Message message) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug(format("Finished [publish] of [publishConsume] to the %s: [%s]. Waiting for reply.",
                            getDestinationType(publishDestination), publishDestination));
      }

      try {
        consumer = consumerFactory.apply(requestReplyWrapper.createSelector(requestReplyPattern, message));
        consumer.consume(consumeParameters.getMaximumWaitUnit().toMillis(consumeParameters.getMaximumWait()), scheduler,
                         new AsyncConsumerCompletionListener());

      } catch (Exception e) {
        String msg = format("An error occurred while listening for the reply from the %s: [%s]: %s",
                            getDestinationType(replyConsumerType), replyDestinationName, e.getMessage());
        completionCallback.error(new JmsConsumeException(msg, e));
      } finally {
        resourceCloserAction.closeResources();
      }
    }

    @Override
    public void onException(Message message, Exception exception) {
      try {
        if (exception instanceof JMSSecurityException) {
          String msg = format("A security error occurred while publishing a message to the %s: [%s]: %s",
                              getDestinationType(publishDestination), publishDestination, exception.getMessage());
          completionCallback.error(new JmsSecurityException(msg, exception));
        } else if (exception instanceof JmsExtensionException) {
          completionCallback.error(exception);
        } else {
          String msg = format("An error occurred while publishing the message for the reply from the %s: [%s]: %s",
                              getDestinationType(publishDestination), publishDestination, exception.getMessage());
          completionCallback.error(new JmsPublishException(msg, exception));
        }
      } finally {
        resourceCloserAction.closeResources();
      }
    }

    private class AsyncConsumerCompletionListener implements CompletionListener {

      @Override
      public void onCompletion(Message received) {
        try {
          if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Received message from %s [%s]. Creating response result", getDestinationType(replyConsumerType),
                         replyDestinationName);
          }

          if (received == null) {
            completionCallback.success(resultFactory.createEmptyResult());
          } else {
            evaluateMessageAck(resolvedAckMode, consumerSession, received, sessionManager, null, null);

            completionCallback.success(resultFactory.createResult(received, connection.getJmsSupport().getSpecification(),
                                                                  resolveOverride(resolveMessageContentType(received, config
                                                                      .getContentType()),
                                                                                  consumeParameters
                                                                                      .getInboundContentType()),
                                                                  resolveOverride(resolveMessageEncoding(received, config
                                                                      .getEncoding()),
                                                                                  consumeParameters.getInboundEncoding()),
                                                                  consumerSession.getAckId()));
          }
        } catch (JMSSecurityException e) {
          String msg = format("A security error occurred while listening for the reply from the %s: [%s]: %s",
                              getDestinationType(replyConsumerType), replyDestinationName, e.getMessage());
          completionCallback.error(new JmsSecurityException(msg, e));
        } catch (Exception e) {
          String msg = format("An error occurred while listening for the reply from the %s: [%s]: %s",
                              getDestinationType(replyConsumerType), replyDestinationName, e.getMessage());
          completionCallback.error(new JmsConsumeException(msg, e));
        } finally {
          releaseConsumerResources();
        }
      }

      @Override
      public void onException(Message message, Exception exception) {
        releaseConsumerResources();
        if (exception instanceof JMSSecurityException) {
          String msg = format("A security error occurred while consuming a message to the %s: [%s]: %s",
                              getDestinationType(publishDestination), publishDestination, exception.getMessage());
          completionCallback.error(new JmsSecurityException(msg, exception));
        } else if (exception instanceof JmsExtensionException) {
          completionCallback.error(exception);
        } else {
          String msg = format("An error occurred while listening for the message reply from the %s: [%s]: %s",
                              getDestinationType(publishDestination), publishDestination, exception.getMessage());
          completionCallback.error(new JmsConsumeException(msg, exception));
        }
      }
    }

    private void releaseConsumerResources() {
      scheduler.submit(() -> {
        closeQuietly(consumer);
        closeQuietly(consumerSession);
        try {
          deleteTemporaryQueue(replyDestination);
        } catch (JMSException e) {
          LOGGER.debug("Unexpected error when trying to delete temporary queue", e);
        }
      });
    }
  }

  private void deleteTemporaryQueue(Destination replyDestination) throws JMSException {
    if (replyDestination instanceof TemporaryQueue) {
      ((TemporaryQueue) replyDestination).delete();
    } else if (replyDestination instanceof TemporaryTopic) {
      ((TemporaryTopic) replyDestination).delete();
    }
  }

  private ConsumerType getConsumerType(Destination replyDestination) {
    ConsumerType replyConsumerType;
    if (replyDestination instanceof Queue) {
      replyConsumerType = new QueueConsumer();
    } else {
      replyConsumerType = new TopicConsumer();
    }
    return replyConsumerType;
  }

  private Destination setReplyDestination(JmsMessageBuilder messageBuilder, JmsSession session,
                                          JmsSupport jmsSupport, Message message)
      throws JMSException {

    if (message.getJMSReplyTo() != null) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug(format("Using provided destination: [%s]", messageBuilder.getReplyTo().getDestination()));
      }
      return message.getJMSReplyTo();
    } else {
      TemporaryQueue temporaryDestination = jmsSupport.createTemporaryDestination(session.get());
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug(format("Using temporary destination: [%s]", ((Queue) temporaryDestination).getQueueName()));
      }
      message.setJMSReplyTo(temporaryDestination);
      return temporaryDestination;
    }
  }

  private String getReplyDestinationName(Destination destination, ConsumerType replyConsumerType) {
    try {
      return replyConsumerType.topic() ? ((Topic) destination).getTopicName() : ((Queue) destination).getQueueName();
    } catch (JMSException e) {
      return destination.toString();
    }
  }

  @Override
  public void dispose() {
    if (scheduler != null) {
      scheduler.stop();
    }
  }
}
