/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.config;

import org.mule.jms.commons.api.config.DefaultJmsConsumerConfig;
import org.mule.jms.commons.api.config.DefaultJmsProducerConfig;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.param.DefaultEncoding;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.ParameterGroup;
import org.mule.runtime.extension.api.annotation.param.display.Placement;
import org.mule.runtime.extension.api.runtime.parameter.OutboundCorrelationStrategy;

import javax.jms.Message;
import javax.jms.Queue;
import javax.jms.Topic;

import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.extension.api.runtime.parameter.OutboundCorrelationStrategy.AUTO;

/**
 * Base configuration for {@link JmsConnector}
 *
 * @since 1.0
 */
public class DefaultJmsConfig {

  @DefaultEncoding
  private String muleEncoding;
  /**
   * The default {@code encoding} of the {@link Message} {@code body} to be used if the message doesn't communicate it
   */
  @Parameter
  @Optional
  @Expression(NOT_SUPPORTED)
  private String encoding;

  /**
   * The default {@code contentType} of the {@link Message} {@code body} to be used if the message doesn't communicate it
   */
  @Parameter
  @Expression(NOT_SUPPORTED)
  @Optional(defaultValue = "*/*")
  private String contentType;

  /**
   * Configuration parameters for consuming {@link Message}s from a JMS {@link Queue} or {@link Topic}
   */
  @Expression(NOT_SUPPORTED)
  @Placement(tab = "Consumer")
  @ParameterGroup(name = "Consumer Config", showInDsl = true)
  private DefaultJmsConsumerConfig consumerConfig;

  /**
   * Configuration parameters for sending {@link Message}s to a JMS {@link Queue} or {@link Topic}
   */
  @Expression(NOT_SUPPORTED)
  @Placement(tab = "Producer")
  @ParameterGroup(name = "Producer Config", showInDsl = true)
  private DefaultJmsProducerConfig producerConfig;

  /**
   * Whether to specify a correlationId when publishing messages. This applies both for custom correlation ids specifies at the
   * operation level and for default correlation Ids taken from the current event
   */
  @Parameter
  @Optional(defaultValue = "AUTO")
  private OutboundCorrelationStrategy sendCorrelationId = AUTO;

  public String getContentType() {
    return contentType;
  }

  public String getEncoding() {
    return encoding == null ? muleEncoding : encoding;
  }

  public DefaultJmsConsumerConfig getConsumerConfig() {
    return consumerConfig;
  }

  public DefaultJmsProducerConfig getProducerConfig() {
    return producerConfig;
  }

}
