/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.connection.provider;

import static java.util.Optional.empty;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import org.mule.jms.commons.internal.connection.param.GenericConnectionParameters;
import org.mule.jms.commons.internal.connection.param.XaPoolParameters;
import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.core.api.MuleContext;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.Optional;

import javax.jms.ConnectionFactory;
import javax.jms.ExceptionListener;

/**
 * Decorator for {@link ConnectionFactory} to intercept method calls and apply logic based on the {@link ConnectionFactory} type
 * and the transactional context in which is being used.
 *
 * @since 1.4
 */
public class InternalConnectionFactoryDecorator implements ConnectionFactoryDecorator {

  private MuleContext muleContext;
  private Registry registry;
  private final Constructor<?> constructor;

  private final Method setReuseSessions;
  private final Method setUserName;
  private final Method setPassword;
  private final Method setMinPoolSize;
  private final Method setMaxPoolSize;
  private final Method setMaxIdleTime;
  private final Method setClientId;
  private final Method setExceptionListener;
  private final Method buildMethod;

  private final Method appliesTo;
  private final Method decorate;

  private final LazyValue<Collection> factoryDecorators;

  InternalConnectionFactoryDecorator(MuleContext muleContext, Registry registry) {
    this.muleContext = muleContext;
    this.registry = registry;

    try {
      Class<?> decoratorClass = Class.forName("com.mulesoft.mule.runtime.bti.api.jms.ConnectionFactoryDecorator");

      Class<?> decoratorBuilderClass = Class.forName("com.mulesoft.mule.runtime.bti.api.jms.JmsConnectionConfig$Builder");
      Class jmsConfigClass = Class.forName("com.mulesoft.mule.runtime.bti.api.jms.JmsConnectionConfig");

      constructor = decoratorBuilderClass.getConstructor(String.class);

      setReuseSessions = decoratorBuilderClass.getMethod("setReuseSessions", boolean.class);
      setUserName = decoratorBuilderClass.getMethod("setUserName", String.class);
      setPassword = decoratorBuilderClass.getMethod("setPassword", String.class);
      setClientId = decoratorBuilderClass.getMethod("setClientId", String.class);
      setExceptionListener = decoratorBuilderClass.getMethod("setExceptionListener", ExceptionListener.class);
      buildMethod = decoratorBuilderClass.getMethod("build");

      // mule versions prior to 4.2.1 and 4.1.6 don't have this, so fetch them using safeGetMethod
      setMinPoolSize = safeGetMethod(decoratorBuilderClass, "setMinPoolSize", int.class).orElse(null);
      setMaxPoolSize = safeGetMethod(decoratorBuilderClass, "setMaxPoolSize", int.class).orElse(null);
      setMaxIdleTime = safeGetMethod(decoratorBuilderClass, "setMaxIdleTime", int.class).orElse(null);

      appliesTo = decoratorClass.getMethod("appliesTo", ConnectionFactory.class, MuleContext.class);
      decorate = decoratorClass.getMethod("decorate", ConnectionFactory.class, jmsConfigClass, MuleContext.class);

      factoryDecorators = new LazyValue<>(() -> {
        try {
          return registry.lookupAllByType(decoratorClass);
        } catch (Exception e) {
          throw new RuntimeException(e);
        }
      });
    } catch (ClassNotFoundException | NoSuchMethodException e) {
      throw new MuleRuntimeException(createStaticMessage("An error occurred trying to load XA Connection Factory Decorator"), e);
    }
  }

  /**
   * Decorates a Connection Factory with the available Connection Factory Decorators on the Mule Runtime.
   *
   * Dev Notes: To avoid the JMS Client require on Bitronix Module to compile, this class uses reflection over the module.
   *
   * @param connectionFactory Connection Factory to decorate
   * @param configName        Global Element name of the JMS Configuration
   * @param reuseSessions     If the sessions should be reused
   * @param username          Username of the JMS Connection
   * @param password          Password of the JMS Connection
   * @param clientId          ClientID of the JMS Connection
   * @param exceptionListener ExceptionListener of the JMS Connection
   *
   * @return A possible decorated ConnectionFactory, the same will be returned in case of no decorator found.
   */

  public ConnectionFactory decorate(ConnectionFactory connectionFactory, String configName, boolean reuseSessions,
                                    GenericConnectionParameters connectionParameters, XaPoolParameters xaPoolParameters,
                                    ExceptionListener exceptionListener) {


    ConnectionFactory decoratedConnectionFactory;
    try {

      Collection<?> decorators = this.factoryDecorators.get();
      decoratedConnectionFactory = connectionFactory;

      if (!decorators.isEmpty()) {
        Object jmsConfigBuilder = constructor.newInstance(configName);
        setReuseSessions.invoke(jmsConfigBuilder, reuseSessions);
        setUserName.invoke(jmsConfigBuilder, connectionParameters.getUsername());
        setPassword.invoke(jmsConfigBuilder, connectionParameters.getPassword());
        setClientId.invoke(jmsConfigBuilder, connectionParameters.getClientId());
        setExceptionListener.invoke(jmsConfigBuilder, exceptionListener);

        configureXAPool(xaPoolParameters, jmsConfigBuilder);

        Object jmsConfig = buildMethod.invoke(jmsConfigBuilder);

        for (Object decorator : decorators) {
          if (appliesTo(appliesTo, decoratedConnectionFactory, decorator)) {
            decoratedConnectionFactory = decorate(decorate, decoratedConnectionFactory, jmsConfig, muleContext, decorator);
          }
        }

      }
    } catch (InstantiationException | IllegalAccessException
        | InvocationTargetException e) {
      throw new RuntimeException(e);
    }

    return decoratedConnectionFactory;
  }

  /**
   * mule versions prior to 4.2.1 and 4.1.6 don't support xa pool configuration. This method keeps in mind that the setters might
   * not be there
   */
  private void configureXAPool(XaPoolParameters xaPoolParameters, Object jmsConfigBuilder)
      throws IllegalAccessException, InvocationTargetException {

    if (setMinPoolSize != null) {
      setMinPoolSize.invoke(jmsConfigBuilder, xaPoolParameters.getMinPoolSize());
    }

    if (setMaxPoolSize != null) {
      setMaxPoolSize.invoke(jmsConfigBuilder, xaPoolParameters.getMaxPoolSize());
    }

    if (setMaxIdleTime != null) {
      setMaxIdleTime.invoke(jmsConfigBuilder, xaPoolParameters.getMaxIdleTime());
    }
  }

  private ConnectionFactory decorate(Method decorate, ConnectionFactory decoratedConnectionFactory, Object build,
                                     MuleContext muleContext, Object decorator) {
    try {
      return (ConnectionFactory) decorate.invoke(decorator, decoratedConnectionFactory, build, muleContext);
    } catch (IllegalAccessException | InvocationTargetException e) {
      throw new MuleRuntimeException(createStaticMessage("An error occurred trying to decorate a Connection Factory."), e);
    }
  }

  private boolean appliesTo(Method appliesTo, ConnectionFactory decoratedConnectionFactory, Object decorator) {
    try {
      return (boolean) appliesTo.invoke(decorator, decoratedConnectionFactory, muleContext);
    } catch (IllegalAccessException | InvocationTargetException e) {
      throw new MuleRuntimeException(createStaticMessage("An error occurred trying to check if the Connection Factory can be decorated."),
                                     e);
    }
  }

  private Optional<Method> safeGetMethod(Class<?> targetClass, String methodName, Class<?>... paramTypes) {
    try {
      return Optional.of(targetClass.getMethod(methodName, paramTypes));
    } catch (NoSuchMethodException e) {
      return empty();
    }
  }
}
