/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.api.message;

import org.mule.jms.commons.api.destination.JmsDestination;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;

import java.io.Serializable;

import javax.jms.DeliveryMode;

/**
 * JMS header fields contain values used by both clients and providers to identify and route messages. A message's complete header
 * is transmitted to all JMS clients that receive the message.
 *
 * @since 1.0
 */
public class JmsHeaders implements Serializable {

  /**
   * The destination from which the JMS Message was received.
   */
  @Parameter
  private JmsDestination destination;

  /**
   * The delivery mode specified when the message was sent, which can be either {@link DeliveryMode#NON_PERSISTENT} with value '1'
   * or {@link DeliveryMode#PERSISTENT} with value '2'
   */
  @Parameter
  private int deliveryMode;

  /**
   * The message's expiration time or {@code zero} if the message does not expire.
   *
   * JMS provider calculates its expiration time by adding the {@code timeToLive} value specified on the send method to the time
   * the message was sent (for transacted sends, this is the time the client sends the message, not the time the transaction is
   * committed)
   * <p>
   * If the {@code timeToLive} is specified as {@code zero}, the message's expiration time is set to zero to indicate that the
   * message does not expire.
   */
  @Parameter
  private long expiration;

  /**
   * The message priority level.
   *
   * JMS defines a ten level priority value with '0' as the lowest priority and '9' as the highest. In addition, clients should
   * consider priorities 0-4 as gradations of {@code normal} priority and priorities 5-9 as gradations of {@code expedited}
   * priority.
   * <p>
   * JMS does not require that a provider strictly implement priority ordering of messages; however, it should do its best to
   * deliver expedited messages ahead of normal messages.
   */
  @Parameter
  private int priority;

  /**
   * A value that uniquely identifies each message sent by a provider.
   *
   * If the Producer implementation supports optimizations and was configured to avoid creating the Message ID using the
   * {@code disableMessageID} parameter, then this header will be {@code null}.
   */
  @Parameter
  @Optional
  private String messageId;

  /**
   * The time a message was handed off to a provider to be sent. It is not the time the message was actually transmitted because
   * the actual send may occur later due to transactions or other client side queueing of messages.
   * <p>
   * If the Producer implementation supports optimizations and was configured to avoid creating the Message timestamp using the
   * {@code disableMessageTimestamp} parameter, then {@code zero} is set.
   */
  @Parameter
  private long timestamp;

  /**
   * The message correlationId, used to link one message with another.
   *
   * A typical use is to link a response message with its request message, using its messageID
   */
  @Parameter
  @Optional
  private String correlationId;

  /**
   * The name of the Destination supplied by a client when a message is sent, where a reply to the message should be sent. If no
   * {@code replyTo} destination was set, then {@code null} is set.
   */
  @Parameter
  @Optional
  private JmsDestination replyTo;

  /**
   * A message type identifier supplied by a client when a message is sent.
   */
  @Parameter
  private String type;

  /**
   * If {@code true}, it is likely, but not guaranteed, that this message was delivered but not acknowledged in the past. Relates
   * to the {@code JMSXDeliveryCount} message property.
   *
   * {@code true} if the message may have been delivered in the past
   */
  @Parameter
  private boolean redelivered;

  /**
   * Present only in JMS 2.0 Messages. It's the message's delivery time or {@code null} if not using JMS 2.0
   *
   * <p>
   * JMS provider calculates its delivery time by adding the {@code deliveryDelay} value specified on the send method to the time
   * the message was sent (for transacted sends, this is the time the client sends the message, not the time the transaction is
   * committed).
   * <p>
   * A message's delivery time is the earliest time when a provider may make the message visible on the target destination and
   * available for delivery to consumers.
   */
  @Parameter
  @Optional
  private Long deliveryTime;

  @Deprecated
  public String getJMSMessageID() {
    return messageId;
  }

  @Deprecated
  public long getJMSTimestamp() {
    return timestamp;
  }

  @Deprecated
  public String getJMSCorrelationID() {
    return correlationId;
  }

  @Deprecated
  public JmsDestination getJMSReplyTo() {
    return replyTo;
  }

  @Deprecated
  public JmsDestination getJMSDestination() {
    return destination;
  }

  @Deprecated
  public int getJMSDeliveryMode() {
    return deliveryMode;
  }

  @Deprecated
  public boolean getJMSRedelivered() {
    return redelivered;
  }

  @Deprecated
  public String getJMSType() {
    return type;
  }

  @Deprecated
  public long getJMSExpiration() {
    return expiration;
  }

  @Deprecated
  public Long getJMSDeliveryTime() {
    return deliveryTime;
  }

  @Deprecated
  public int getJMSPriority() {
    return priority;
  }

  public static class Builder implements JmsHeadersBuilder {

    private JmsHeaders jmsHeaders = new JmsHeaders();

    @Override
    public JmsHeaders.Builder setMessageId(String messageId) {
      jmsHeaders.messageId = messageId;
      return this;
    }

    @Override
    public JmsHeaders.Builder setTimestamp(long timestamp) {
      jmsHeaders.timestamp = timestamp;
      return this;
    }

    @Override
    public JmsHeaders.Builder setCorrelationId(String correlationId) {
      jmsHeaders.correlationId = correlationId;
      return this;
    }

    @Override
    public JmsHeaders.Builder setReplyTo(JmsDestination replyTo) {
      jmsHeaders.replyTo = replyTo;
      return this;
    }

    @Override
    public JmsHeaders.Builder setDestination(JmsDestination destination) {
      jmsHeaders.destination = destination;
      return this;
    }

    @Override
    public JmsHeaders.Builder setDeliveryMode(int deliveryMode) {
      jmsHeaders.deliveryMode = deliveryMode;
      return this;
    }

    @Override
    public JmsHeaders.Builder setRedelivered(boolean redelivered) {
      jmsHeaders.redelivered = redelivered;
      return this;
    }

    @Override
    public JmsHeaders.Builder setType(String type) {
      jmsHeaders.type = type;
      return this;
    }

    @Override
    public JmsHeaders.Builder setExpiration(long expiration) {
      jmsHeaders.expiration = expiration;
      return this;
    }

    @Override
    public JmsHeaders.Builder setPriority(int priority) {
      jmsHeaders.priority = priority;
      return this;
    }

    @Override
    public JmsHeaders.Builder setDeliveryTime(long deliveryTime) {
      jmsHeaders.deliveryTime = deliveryTime;
      return this;
    }

    @Override
    public JmsHeaders build() {
      return jmsHeaders;
    }
  }

  public JmsDestination getDestination() {
    return destination;
  }

  public void setDestination(JmsDestination destination) {
    this.destination = destination;
  }

  public int getDeliveryMode() {
    return deliveryMode;
  }

  public void setDeliveryMode(int deliveryMode) {
    this.deliveryMode = deliveryMode;
  }

  public long getExpiration() {
    return expiration;
  }

  public void setExpiration(long expiration) {
    this.expiration = expiration;
  }

  public int getPriority() {
    return priority;
  }

  public void setPriority(int priority) {
    this.priority = priority;
  }

  public String getMessageId() {
    return messageId;
  }

  public void setMessageId(String messageId) {
    this.messageId = messageId;
  }

  public long getTimestamp() {
    return timestamp;
  }

  public void setTimestamp(long timestamp) {
    this.timestamp = timestamp;
  }

  public String getCorrelationId() {
    return correlationId;
  }

  public void setCorrelationId(String correlationId) {
    this.correlationId = correlationId;
  }

  public JmsDestination getReplyTo() {
    return replyTo;
  }

  public void setReplyTo(JmsDestination replyTo) {
    this.replyTo = replyTo;
  }

  public String getType() {
    return type;
  }

  public void setType(String type) {
    this.type = type;
  }

  public boolean isRedelivered() {
    return redelivered;
  }

  public void setRedelivered(boolean redelivered) {
    this.redelivered = redelivered;
  }

  public Long getDeliveryTime() {
    return deliveryTime;
  }

  public void setDeliveryTime(Long deliveryTime) {
    this.deliveryTime = deliveryTime;
  }
}
