/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.connection.exception;

import org.mule.jms.commons.internal.connection.IBMJmsCachingConnectionFactory;
import org.mule.jms.commons.internal.connection.JmsConnection;
import org.mule.jms.commons.internal.source.JmsListener;
import org.slf4j.Logger;

import javax.jms.ExceptionListener;
import javax.jms.JMSException;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import static org.slf4j.LoggerFactory.getLogger;

/**
 * Composite pattern implementation for {@link ExceptionListener}. This gives the capability of register multiple
 * {@link ExceptionListener} for the same exception notification, this is useful to be able to notify all the {@link JmsListener}
 * that has been created with the same {@link JmsConnection}.
 *
 * @since 1.0
 */
public class CompositeJmsExceptionListener implements ExceptionListener {

  private static final Logger LOGGER = getLogger(CompositeJmsExceptionListener.class);
  private List<ExceptionListener> exceptionListenerList = new CopyOnWriteArrayList<>();

  /**
   * {@inheritDoc}
   */
  @Override
  public void onException(JMSException exception) {
    int exceptionListenersCount = exceptionListenerList.size();
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Length of exceptionListenerList: {}", exceptionListenersCount);
    }
    exceptionListenerList.forEach(listener -> listener.onException(exception));
  }

  /**
   * Registers a {@link ExceptionListener} to be notified if an connection error occurs
   * 
   * @param exceptionListener The listener to register
   */
  public void registerExceptionListener(ExceptionListener exceptionListener) {
    exceptionListenerList.add(exceptionListener);
  }
}
