/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.operation;

import static java.lang.String.format;
import static org.mule.jms.commons.internal.common.JmsCommons.isManualAck;
import static org.mule.jms.commons.internal.common.JmsCommons.createJmsSession;
import static org.mule.jms.commons.internal.common.JmsCommons.evaluateMessageAck;
import static org.mule.jms.commons.internal.common.JmsCommons.getDestinationType;
import static org.mule.jms.commons.internal.common.JmsCommons.isPartOfCurrentTx;
import static org.mule.jms.commons.internal.common.JmsCommons.releaseResources;
import static org.mule.jms.commons.internal.common.JmsCommons.resolveMessageContentType;
import static org.mule.jms.commons.internal.common.JmsCommons.resolveMessageEncoding;
import static org.mule.jms.commons.internal.common.JmsCommons.resolveOverride;
import static org.mule.jms.commons.internal.common.JmsCommons.toInternalAckMode;
import static org.mule.jms.commons.internal.config.InternalAckMode.AUTO;
import static org.mule.jms.commons.internal.config.InternalAckMode.DUPS_OK;
import static org.mule.jms.commons.internal.config.InternalAckMode.IMMEDIATE;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.jms.commons.api.config.JmsConsumerConfig;
import org.mule.jms.commons.api.destination.ConsumerType;
import org.mule.jms.commons.api.exception.JmsConsumeException;
import org.mule.jms.commons.api.exception.JmsExtensionException;
import org.mule.jms.commons.api.exception.JmsSecurityException;
import org.mule.jms.commons.api.message.JmsAttributes;
import org.mule.jms.commons.internal.config.InternalAckMode;
import org.mule.jms.commons.internal.config.JmsAckMode;
import org.mule.jms.commons.internal.config.JmsConfig;
import org.mule.jms.commons.internal.connection.JmsTransactionalConnection;
import org.mule.jms.commons.internal.connection.session.JmsMessageAckedMonitor;
import org.mule.jms.commons.internal.connection.session.JmsSession;
import org.mule.jms.commons.internal.connection.session.JmsSessionManager;
import org.mule.jms.commons.internal.consume.JmsMessageConsumer;
import org.mule.jms.commons.internal.message.JmsResultFactory;
import org.mule.jms.commons.internal.support.JmsSupport;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;
import org.mule.runtime.extension.api.tx.OperationTransactionalAction;

import java.util.concurrent.TimeUnit;

import javax.jms.CompletionListener;
import javax.jms.Destination;
import javax.jms.IllegalStateException;
import javax.jms.JMSException;
import javax.jms.JMSSecurityException;
import javax.jms.Message;

import org.slf4j.Logger;

/**
 * Operation that allows the user to consume a single {@link Message} from a given {@link Destination}
 *
 * @since 1.0
 */
public class JmsConsume implements Disposable {

  private static final Logger LOGGER = getLogger(JmsConsume.class);

  private static final String IBM_MQRC_CONNECTION_BROKEN = "MQRC_CONNECTION_BROKEN";
  private static final String AMQ_NOT_CONNECTED_ERROR = "AMQ219010";
  private static final String LOST_SERVER_EXCEPTION = "LostServerException";

  private static final String IBM_OPEN_QUEUE_FAILED = "JMSWMQ2008";
  private static final String IBM_CHANNEL_CLOSED_CODE = "JMSWMQ0018";

  private final JmsResultFactory resultFactory = JmsResultFactory.getInstance();
  private final Scheduler scheduler;


  public JmsConsume(JmsSessionManager sessionManager, SchedulerService schedulerService) {
    this.sessionManager = sessionManager;
    this.scheduler = schedulerService.ioScheduler();
  }

  private final JmsSessionManager sessionManager;

  public Result<Object, JmsAttributes> consume(JmsConfig config,
                                               JmsTransactionalConnection connection,
                                               String destination,
                                               ConsumerType consumerType,
                                               JmsAckMode ackMode,
                                               String selector,
                                               String contentType,
                                               String encoding,
                                               Long maximumWait,
                                               TimeUnit maximumWaitUnit,
                                               OperationTransactionalAction transactionalAction)
      throws JmsExtensionException, ConnectionException {

    InternalAckMode resolvedAckMode = resolveAck(config.getConsumerConfig(), ackMode);
    Result<Object, JmsAttributes> result = null;
    JmsSession session = null;
    boolean partOfCurrentTx = false;
    JmsMessageConsumer consumer = null;
    try {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Begin [consume] on the " + getDestinationType(consumerType) + ": ["
            + destination + "]");
      }

      JmsSupport jmsSupport = connection.getJmsSupport();
      session = createJmsSession(connection, resolvedAckMode, consumerType.topic(), sessionManager, transactionalAction);
      partOfCurrentTx = isPartOfCurrentTx(session, connection, sessionManager);
      Destination jmsDestination = jmsSupport.createDestination(session.get(), destination, consumerType.topic(), config);

      consumer = connection.createConsumer(session, jmsDestination, selector, consumerType);

      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Consuming Message from the " + getDestinationType(consumerType) + ": ["
            + destination + "]");
      }
      final JmsSession finalSession = session;
      final boolean finalPartOfCurrentTx = partOfCurrentTx;
      final JmsMessageConsumer finalConsumer = consumer;
      final JmsMessageAckedMonitor jmsMessageAckedMonitor = new JmsMessageAckedMonitor();
      final Message message = consumer.consume(maximumWaitUnit.toMillis(maximumWait));
      if (message == null) {
        result = resultFactory.createEmptyResult();
      } else {

        String resolvedContentType =
            resolveOverride(resolveMessageContentType(message, config.getContentType()), contentType);
        String resolvedEncoding = resolveOverride(resolveMessageEncoding(message, config.getEncoding()), encoding);

        result = resultFactory.createResult(message, jmsSupport.getSpecification(),
                                            resolvedContentType, resolvedEncoding,
                                            session.getAckId());

        evaluateMessageAck(resolvedAckMode, session, message, sessionManager, null, jmsMessageAckedMonitor);
      }
      if (isManualAck(finalSession) && sessionManager.isPendingAck(finalSession.getAckId().get())) {
        config.getResourceReleaserScheduler().execute(() -> {
          jmsMessageAckedMonitor.waitForMessageAcked();
          releaseResources(finalSession, finalPartOfCurrentTx, finalConsumer);
        });
      } else {
        releaseResources(finalSession, finalPartOfCurrentTx, finalConsumer);
      }
    } catch (JMSSecurityException e) {
      String msg = format("A security error occurred while consuming a message from the %s: [%s]: %s",
                          getDestinationType(consumerType), destination, e.getMessage());
      releaseResources(session, partOfCurrentTx, consumer);
      throw new JmsSecurityException(msg, e);
    } catch (IllegalStateException e) {
      boolean isConnectionError = session == null;
      String msg = format((isConnectionError ? "A connection error" : "An error")
          + " occurred while consuming a message from the %s: [%s]: %s",
                          getDestinationType(consumerType), destination, e.getMessage());
      if (e.getMessage().contains(IBM_MQRC_CONNECTION_BROKEN)) {
        isConnectionError = true;
      }
      releaseResources(session, partOfCurrentTx, consumer);
      if (isConnectionError) {
        throw new ConnectionException(e);
      } else {
        throw new JmsConsumeException(msg, e);
      }
    } catch (JmsConsumeException e) {
      releaseResources(session, partOfCurrentTx, consumer);
      throw e;
    } catch (JMSException e) {
      boolean isConnectionError = false;
      // It was necessary to add these conditions for reconnect when Artemis or Weblogic are disconnected
      if (e.getMessage() != null && e.getMessage().contains(AMQ_NOT_CONNECTED_ERROR)) {
        isConnectionError = true;
      } else if (e.getCause() != null && e.getCause().getClass().getName().contains(LOST_SERVER_EXCEPTION)) {
        isConnectionError = true;
      }
      releaseResources(session, partOfCurrentTx, consumer);
      if (isConnectionError) {
        String msg = format("A connection error occurred while consuming a message from the %s: [%s]: %s",
                            getDestinationType(consumerType), destination, e.getMessage());
        throw new ConnectionException(msg, e);
      } else {
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("[consume-1] Exception: ", e);
        }
        String msg = format("An error occurred while consuming a message from the %s [%s]: %s",
                            getDestinationType(consumerType), destination, e.getMessage());
        if (isConnectionException(e)) {
          LOGGER.info("[consume-1]: " + msg);
          if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("[consume-1] Triggering reconnection.");
          }
          throw new ConnectionException(e, connection);
        } else {
          throw new JmsConsumeException(msg, e);
        }
      }

    } catch (Exception e) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("[consume-2] Exception: ", e);
      }
      String msg = format("An error occurred while consuming a message from the %s [%s]: %s",
                          getDestinationType(consumerType), destination, e.getMessage());
      releaseResources(session, partOfCurrentTx, consumer);
      if (isConnectionException(e)) {
        LOGGER.info("[consume-2]: " + msg);
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("[consume-2] Triggering reconnection.");
        }
        throw new ConnectionException(e, connection);
      } else {
        throw new JmsConsumeException(msg, e);
      }
    }
    return result;
  }

  public void consume(JmsConfig config,
                      JmsTransactionalConnection connection,
                      String destination,
                      ConsumerType consumerType,
                      JmsAckMode ackMode,
                      String selector,
                      String contentType,
                      String encoding,
                      Long maximumWait,
                      TimeUnit maximumWaitUnit,
                      OperationTransactionalAction transactionalAction,
                      CompletionCallback<Object, JmsAttributes> completionCallback)
      throws JmsExtensionException {

    InternalAckMode resolvedAckMode = resolveAck(config.getConsumerConfig(), ackMode);
    JmsSession session = null;
    boolean partOfCurrentTx = false;
    JmsMessageConsumer consumer = null;
    try {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Begin [consume] on the " + getDestinationType(consumerType) + ": ["
            + destination + "]");
      }

      JmsSupport jmsSupport = connection.getJmsSupport();
      session = createJmsSession(connection, resolvedAckMode, consumerType.topic(), sessionManager, transactionalAction);
      partOfCurrentTx = isPartOfCurrentTx(session, connection, sessionManager);
      Destination jmsDestination = jmsSupport.createDestination(session.get(), destination, consumerType.topic(), config);

      consumer = connection.createConsumer(session, jmsDestination, selector, consumerType);

      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Consuming Message from the " + getDestinationType(consumerType) + ": ["
            + destination + "]");
      }

      JmsMessageConsumer finalConsumer = consumer;
      JmsSession finalSession = session;
      boolean finalPartOfCurrentTx = partOfCurrentTx;

      consumer.consume(maximumWaitUnit.toMillis(maximumWait), new CompletionListener() {

        final JmsMessageAckedMonitor jmsMessageAckedMonitor = new JmsMessageAckedMonitor();

        @Override
        public void onCompletion(Message received) {
          try {
            // If no explicit content type was provided to the operation, fallback to the
            // one communicated in the message properties. Finally if no property was set,
            // use the default one provided by the config

            if (received == null) {
              completionCallback.success(resultFactory.createEmptyResult());
            } else {
              String resolvedContentType =
                  resolveOverride(resolveMessageContentType(received, config.getContentType()), contentType);
              String resolvedEncoding = resolveOverride(resolveMessageEncoding(received, config.getEncoding()), encoding);

              Result<Object, JmsAttributes> result = resultFactory.createResult(received, jmsSupport.getSpecification(),
                                                                                resolvedContentType, resolvedEncoding,
                                                                                finalSession.getAckId());
              evaluateMessageAck(resolvedAckMode, finalSession, received, sessionManager, null, jmsMessageAckedMonitor);

              completionCallback.success(result);
            }

          } catch (Exception e) {
            if (LOGGER.isDebugEnabled()) {
              LOGGER.debug("[consume-3] Exception: ", e);
            }
            String msg = format("An error occurred while consuming a message from the %s [%s]: %s",
                                getDestinationType(consumerType), destination, e.getMessage());
            if (isConnectionException(e)) {
              LOGGER.info("[consume-3]: " + msg);
              if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("[consume-3] Triggering reconnection.");
              }
              completionCallback.error(new ConnectionException(msg, e));
            } else {
              completionCallback.error(new JmsConsumeException(msg, e));
            }
          } finally {
            if (isManualAck(finalSession) && sessionManager.isPendingAck(finalSession.getAckId().get())) {
              config.getResourceReleaserScheduler().execute(() -> {
                jmsMessageAckedMonitor.waitForMessageAcked();
                releaseResources(finalSession, finalPartOfCurrentTx, finalConsumer);
              });
            } else {
              releaseResources(finalSession, finalPartOfCurrentTx, finalConsumer);
            }
          }
        }

        @Override
        public void onException(Message message, Exception e) {
          releaseResources(finalSession, finalPartOfCurrentTx, finalConsumer);
          if (e instanceof JMSSecurityException) {
            String msg = format("A security error occurred while consuming a message from the %s: [%s]: %s",
                                getDestinationType(consumerType), destination, e.getMessage());
            completionCallback.error(new JmsSecurityException(msg, e));
          } else if (e instanceof JmsConsumeException) {
            completionCallback.error(e);
          } else if (e instanceof IllegalStateException) {
            if (e.getMessage().contains(IBM_MQRC_CONNECTION_BROKEN))
              completionCallback.error(new ConnectionException(e));
            else
              completionCallback.error(e);
          } else {
            if (LOGGER.isDebugEnabled()) {
              LOGGER.debug("[consume-4] Exception: ", e);
            }
            String msg = format("An error occurred while consuming a message from the %s [%s]: %s",
                                getDestinationType(consumerType), destination, e.getMessage());
            if (isConnectionException(e)) {
              LOGGER.info("[consume-4]: " + msg);
              if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("[consume-4] Triggering reconnection.");
              }
              completionCallback.error(new ConnectionException(msg, e));
            } else {
              completionCallback.error(new JmsConsumeException(msg, e));
            }
          }
        }
      });
    } catch (JMSSecurityException e) {
      String msg = format("A security error occurred while consuming a message from the %s: [%s]: %s",
                          getDestinationType(consumerType), destination, e.getMessage());
      releaseResources(session, partOfCurrentTx, consumer);
      completionCallback.error(new JmsSecurityException(msg, e));
    } catch (IllegalStateException e) {
      boolean isConnectionError = session == null;
      String msg = format((isConnectionError ? "A connection error" : "An error")
          + " occurred while consuming a message from the %s: [%s]: %s",
                          getDestinationType(consumerType), destination, e.getMessage());
      if (e.getMessage().contains(IBM_MQRC_CONNECTION_BROKEN)) {
        isConnectionError = true;
      }
      releaseResources(session, partOfCurrentTx, consumer);
      completionCallback.error(isConnectionError ? new ConnectionException(e) : new JmsConsumeException(msg, e));
    } catch (Exception e) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("[consume-5] Exception: ", e);
      }
      String msg = format("An error occurred while consuming a message from the %s [%s]: %s",
                          getDestinationType(consumerType), destination, e.getMessage());
      releaseResources(session, partOfCurrentTx, consumer);
      if (isConnectionException(e)) {
        LOGGER.info("[consume-5]: " + msg);
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("[consume-5] Triggering reconnection.");
        }
        completionCallback.error(new ConnectionException(msg, e));
      } else {
        completionCallback.error(new JmsConsumeException(msg, e));
      }
    }
  }

  private InternalAckMode resolveAck(JmsConsumerConfig config, JmsAckMode ackMode) {
    InternalAckMode fallbackAck = toInternalAckMode(config.getAckMode());
    if (AUTO.equals(fallbackAck) || DUPS_OK.equals(fallbackAck)) {
      fallbackAck = IMMEDIATE;
    }
    return resolveOverride(fallbackAck, toInternalAckMode(ackMode));
  }

  public boolean isConnectionException(Exception e) {

    if (!(e instanceof JMSException)) {
      String msg = format("class: %s, errMsg:%s", e.getClass().toString(), e.getMessage());
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("[isConnectionException] false for " + msg);
      }
      return false;
    }
    // handle JMS exceptions
    JMSException ex = ((JMSException) e);
    String errCode = ex.getErrorCode();
    String msg = format("class: %s, errCode: %s, errMsg:%s, linkedEx: %s", ex.getClass().toString(), errCode, ex.getMessage(),
                        ex.getLinkedException());
    if ((errCode != null)
        && ((ex.getClass().toString().contains("DetailedJMSException") && errCode.equals(IBM_CHANNEL_CLOSED_CODE))
            || errCode.equals(IBM_OPEN_QUEUE_FAILED))) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("[isConnectionException] true for " + msg);
      }
      return true;
    }
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("[isConnectionException] false for " + msg);
    }
    return false;
  }

  @Override
  public void dispose() {
    if (scheduler != null) {
      scheduler.stop();
    }
  }
}
