/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.source;

import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.message.Error;
import org.mule.runtime.extension.api.runtime.source.SourceCallbackContext;

/**
 * Strategy interface for different implementations of a JMS Message Consumer
 *
 * @since 1.0.0
 */
public interface MessageConsumerDelegate {

  /**
   * Creates and start N consumers, where N is defined by the param {@code numberOfConsumers}.
   *
   * @param numberOfConsumers number of consumers to create and start.
   * @throws ConnectionException If an error occurs trying to create and start the consumers.
   */
  void createConsumers(int numberOfConsumers) throws ConnectionException;

  /**
   * Communicates when a message has been successfully processed in a flow.
   *
   * @param callbackContext Context with references of the processed message.
   */
  void onSuccess(SourceCallbackContext callbackContext);

  /**
   * Communicates when a message has not been successfully processed in a flow.
   *
   * @param callbackContext Context with references of the processed message.
   * @param error           Produced error on the flow.
   */
  void onError(SourceCallbackContext callbackContext, Error error);

  /**
   * Request to stop all the message consumers.
   */
  void stop();

  void disableConsumers();
}
