/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.operation;

import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.jms.commons.internal.connection.session.AckCallback;
import org.mule.jms.commons.internal.connection.session.JmsSessionManager;
import org.mule.jms.commons.api.config.AckMode;
import org.mule.jms.commons.internal.connection.JmsConnection;
import org.mule.jms.commons.internal.connection.session.JmsSession;
import org.mule.jms.commons.api.exception.JmsAckErrorTypeProvider;
import org.mule.jms.commons.api.exception.JmsAckException;
import org.mule.jms.commons.api.exception.JmsSessionRecoverErrorTypeProvider;
import org.mule.jms.commons.api.exception.JmsSessionRecoverException;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;

import org.slf4j.Logger;

import javax.jms.JMSException;
import javax.jms.Message;


/**
 * Operation that allows the user to perform an ACK over a {@link Message} produced by the current {@link JmsSession}
 *
 * @since 1.0
 */
public final class JmsAcknowledge {

  private static final Logger LOGGER = getLogger(JmsAcknowledge.class);
  private JmsSessionManager sessionManager;

  public JmsAcknowledge(JmsSessionManager sessionManager) {
    this.sessionManager = sessionManager;
  }

  /**
   * Allows the user to perform an ACK when the {@link AckMode#MANUAL} mode is elected while consuming the {@link Message}. As per
   * JMS Spec, performing an ACK over a single {@link Message} automatically works as an ACK for all the {@link Message}s produced
   * in the same {@link JmsSession}.
   *
   * @param ackId The AckId of the Message to ACK
   * @throws JmsAckException if the {@link JmsSession} or {@link JmsConnection} were closed, or if the ID doesn't belong to a
   *                         session of the current connection
   */
  @Throws(JmsAckErrorTypeProvider.class)
  public void ack(@Summary("The AckId of the Message to ACK") String ackId, CompletionCallback<Void, Void> completionCallback) {
    try {
      checkArgument(!isBlank(ackId), "The AckId can not be null or empty");
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Performing ACK on session: " + ackId);
      }

      sessionManager.ack(ackId, new CompletionCallbackAdapter(completionCallback));

    } catch (Exception e) {
      LOGGER.error(format("An error occurred while acking a message with ID [%s]: ", ackId), e);
      completionCallback
          .error(new JmsAckException(format("An error occurred while trying to perform an ACK on Session with ID [%s]: ", ackId),
                                     e));
    }
  }

  /**
   * Allows the user to perform a session recover when the {@link AckMode#MANUAL} mode is elected while consuming the
   * {@link Message}. As per JMS Spec, performing a session recover automatically will redeliver all the consumed messages that
   * had not being acknowledged before this recover.
   *
   * @param ackId The AckId of the Message Session to recover
   */
  @Throws(JmsSessionRecoverErrorTypeProvider.class)
  public void recoverSession(String ackId, CompletionCallback<Void, Void> completionCallback) {
    try {
      checkArgument(!isBlank(ackId), "The AckId can not be null or empty");
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Recovering session: " + ackId);
      }

      sessionManager.recoverSession(ackId, new CompletionCallbackAdapter(completionCallback));

    } catch (Exception e) {
      LOGGER.error(format("An error occurred while recovering the session with ID [%s]: ", ackId), e);
      throw new JmsSessionRecoverException(format("An error occurred while trying to perform an recover on Session with ID [%s]: ",
                                                  ackId),
                                           e);
    }
  }

  private static class CompletionCallbackAdapter implements AckCallback {

    private static final Result<Void, Void> VOID_RESULT = Result.<Void, Void>builder().build();
    private CompletionCallback<Void, Void> completionCallback;

    CompletionCallbackAdapter(CompletionCallback<Void, Void> completionCallback) {
      this.completionCallback = completionCallback;
    }

    @Override
    public void onSuccess() {
      completionCallback.success(VOID_RESULT);
    }

    @Override
    public void onError(JMSException e) {
      completionCallback.error(e);
    }
  }

}
