/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.support;

import static java.util.Optional.empty;
import static javax.jms.DeliveryMode.NON_PERSISTENT;
import static javax.jms.DeliveryMode.PERSISTENT;
import static javax.jms.Session.SESSION_TRANSACTED;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.mule.jms.commons.api.connection.JmsSpecification.JMS_2_0;
import static org.mule.jms.commons.api.connection.LookupJndiDestination.ALWAYS;
import static org.mule.jms.commons.api.connection.LookupJndiDestination.NEVER;
import static org.mule.jms.commons.api.connection.LookupJndiDestination.TRY_ALWAYS;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.jms.commons.api.connection.JmsSpecification;
import org.mule.jms.commons.api.connection.LookupJndiDestination;
import org.mule.jms.commons.api.destination.ConsumerType;
import org.mule.jms.commons.api.destination.TopicConsumer;
import org.mule.jms.commons.api.exception.DestinationNotFoundException;
import org.mule.jms.commons.internal.config.JmsConfig;

import java.util.Optional;
import java.util.function.Function;

import javax.jms.CompletionListener;
import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageProducer;
import javax.jms.Queue;
import javax.jms.Session;
import javax.jms.TemporaryQueue;
import javax.jms.Topic;
import javax.jms.XAConnection;
import javax.jms.XAConnectionFactory;
import javax.jms.XASession;

import org.slf4j.Logger;

/**
 * <code>Jms20Support</code> is a template class to provide an abstraction to to the JMS 2.0 API specification.
 *
 * @since 1.0
 */
public class Jms20Support implements JmsSupport {

  private final static Logger LOGGER = getLogger(Jms20Support.class);

  private final Function<String, Optional<Destination>> jndiObjectSupplier;
  private final LookupJndiDestination lookupJndiDestination;

  public Jms20Support() {
    this(NEVER, (name) -> Optional.empty());
  }

  public Jms20Support(LookupJndiDestination lookupJndiDestination, Function<String, Optional<Destination>> jndiObjectSupplier) {
    checkArgument(lookupJndiDestination != null, "The LookupJndiDestination cannot be null");
    checkArgument(jndiObjectSupplier != null, "The JndiObjectSupplier cannot be null");
    this.lookupJndiDestination = lookupJndiDestination;
    this.jndiObjectSupplier = jndiObjectSupplier;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JmsSpecification getSpecification() {
    return JMS_2_0;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Connection createConnection(ConnectionFactory connectionFactory, String username, String password)
      throws JMSException {
    checkArgument(connectionFactory != null, "connectionFactory cannot be null");
    LOGGER.trace("Creating Connection with username and password");
    return connectionFactory.createConnection(username, password);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public XAConnection createXaConnection(XAConnectionFactory connectionFactory) throws JMSException {
    checkArgument(connectionFactory != null, "connectionFactory cannot be null");
    LOGGER.trace("Creating XaConnection");
    return connectionFactory.createXAConnection();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public XAConnection createXaConnection(XAConnectionFactory connectionFactory, String username, String password)
      throws JMSException {
    checkArgument(connectionFactory != null, "connectionFactory cannot be null");
    LOGGER.trace("Creating XaConnection with UserName and Password");
    return connectionFactory.createXAConnection(username, password);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Connection createConnection(ConnectionFactory connectionFactory) throws JMSException {
    if (connectionFactory == null) {
      throw new IllegalArgumentException("connectionFactory cannot be null");
    }
    LOGGER.trace("Creating connection");
    return connectionFactory.createConnection();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Session createSession(Connection connection, boolean topic, boolean transacted, int ackMode)
      throws JMSException {
    LOGGER.trace("Creating session. Topic [{}],Transacted [{}], AckMode [{}]", topic, transacted, ackMode);
    return connection.createSession(transacted, (transacted ? SESSION_TRANSACTED : ackMode));
  }

  @Override
  public XASession createXaSession(XAConnection connection) throws JMSException {
    checkArgument(connection != null, "XA Connection cannot be null");
    LOGGER.trace("Creating XA Session");
    return connection.createXASession();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MessageProducer createProducer(Session session, Destination destination, boolean topic)
      throws JMSException {
    LOGGER.trace("Creating producer for destination [{}].Is Topic: [{}].", destination, topic);
    return session.createProducer(destination);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MessageConsumer createConsumer(Session session, Destination destination, String messageSelector, ConsumerType type)
      throws JMSException {

    if (!type.topic()) {
      LOGGER.trace("Creating Queue Consumer for [{}] with selector [{}]", destination, messageSelector);
      return session.createConsumer(destination, messageSelector);
    }

    TopicConsumer topicConsumer = (TopicConsumer) type;
    // Shared Consumer exists in JMS 2.0 only
    if (topicConsumer.isDurable() && topicConsumer.isShared()) {
      LOGGER.trace("Creating Shared Durable Topic Consumer for [{}] with Subscription Name [{}] and Message Selector [{}]",
                   destination, topicConsumer.getSubscriptionName(), messageSelector);
      return session.createSharedDurableConsumer((Topic) destination, topicConsumer.getSubscriptionName(), messageSelector);
    }

    if (topicConsumer.isDurable()) {
      LOGGER
          .trace("Creating Durable Topic Subscriber for [{}] with SubscriptionName [{}] and Message Selector [{}]. Is No Local: [{}]",
                 destination, topicConsumer.getSubscriptionName(), messageSelector, topicConsumer.isNoLocal());
      return session.createDurableSubscriber((Topic) destination, topicConsumer.getSubscriptionName(), messageSelector,
                                             topicConsumer.isNoLocal());
    }

    if (topicConsumer.isShared()) {
      LOGGER.trace("Creating Shared Topic Consumer for [{}] with SubscriptionName [{}] and Message Selector [{}]", destination,
                   topicConsumer.getSubscriptionName(), messageSelector);
      return session.createSharedConsumer((Topic) destination, topicConsumer.getSubscriptionName(), messageSelector);
    }

    LOGGER.trace("Creating consumer for Topic [{}] with Message Selector [{}]. Is No Local: [{}]", destination, messageSelector,
                 topicConsumer.isNoLocal());
    return session.createConsumer(destination, messageSelector, topicConsumer.isNoLocal());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Destination createDestination(Session session, String name, boolean topic, JmsConfig config)
      throws JMSException, DestinationNotFoundException {

    if (getLookupJndiDestination().equals(ALWAYS) || getLookupJndiDestination().equals(TRY_ALWAYS)) {
      Optional<Destination> destination = createDestinationFromJndi(name);
      if (destination.isPresent()) {
        return destination.get();

      } else if (getLookupJndiDestination().equals(ALWAYS)) {

        LOGGER.debug("Failed to find JNDI destination [{}], but destination origin was forced as ALWAYS use JNDI."
            + " We have to stop execution.", name);

        throw new DestinationNotFoundException("Failed to find JNDI destination [" + name + "]");
      }
    }

    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Using non-JNDI destination [{}], will create one now", name);
    }

    checkArgument(session != null, "Session cannot be null when creating a destination");
    checkArgument(!isBlank(name), "Destination name cannot be blank when creating a destination");

    Destination destination;

    if (topic) {
      LOGGER.debug("Creating Topic Destination with name: [{}]", name);
      destination = session.createTopic(name);
    } else {
      LOGGER.debug("Creating Queue Destination with name: [{}]", name);
      destination = session.createQueue(name);
    }

    return destination;
  }

  protected Optional<Destination> createDestinationFromJndi(String name) throws JMSException {
    Optional<Destination> dest = getJndiDestination(name);

    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Destination [{}] {}", name, dest.isPresent() ? "located in JNDI, will use it now" : "not found using JNDI");
    }

    return dest;
  }

  protected Optional<Destination> getJndiDestination(String name) {
    try {
      LOGGER.debug("Looking up {} from JNDI", name);
      return getJndiObjectSupplier().apply(name);
    } catch (Exception e) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Failed to look up destination [{}]: ", name, e);
      }

      return empty();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TemporaryQueue createTemporaryDestination(Session session) throws JMSException {
    checkArgument(session != null, "Session cannot be null when creating a destination");

    LOGGER.debug("Creating temporary destination");
    return session.createTemporaryQueue();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void send(MessageProducer producer, Message message, boolean persistent, int priority,
                   long ttl, boolean topic)
      throws JMSException {
    if (LOGGER.isDebugEnabled()) {
      Destination dest = producer.getDestination();
      LOGGER.debug("Sending message to [{}], persistent:[{}], with priority:[{}] and ttl:[{}]",
                   dest instanceof Queue ? ((Queue) dest).getQueueName() : ((Topic) dest).getTopicName(),
                   persistent, priority, ttl);
    }
    producer.send(message, (persistent ? PERSISTENT : NON_PERSISTENT), priority, ttl);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void send(MessageProducer producer, Message message, boolean persistent, int priority, long ttl, boolean topic,
                   CompletionListener completionListener)
      throws JMSException {
    if (LOGGER.isDebugEnabled()) {
      Destination dest = producer.getDestination();
      LOGGER.debug("Sending message to [{}], persistent:[{}], with priority:[{}] and ttl:[{}]",
                   dest instanceof Queue ? ((Queue) dest).getQueueName() : ((Topic) dest).getTopicName(),
                   persistent, priority, ttl);
    }
    producer.send(message, (persistent ? PERSISTENT : NON_PERSISTENT), priority, ttl, completionListener);
  }

  private Function<String, Optional<Destination>> getJndiObjectSupplier() {
    return jndiObjectSupplier;
  }

  protected LookupJndiDestination getLookupJndiDestination() {
    return lookupJndiDestination;
  }
}
