/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.common;

import static java.lang.String.format;
import static org.mule.jms.commons.internal.common.JmsCommons.releaseResources;
import static org.mule.jms.commons.internal.publish.PublishConnectionExceptionErrorCodes.DETAILED_JMS_EXCEPTION_CODES;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.jms.commons.api.exception.JmsExtensionException;
import org.mule.jms.commons.api.exception.JmsPublishException;
import org.mule.jms.commons.api.exception.JmsSecurityException;
import org.mule.jms.commons.internal.connection.session.JmsSession;
import org.mule.jms.commons.internal.operation.JmsPublish;
import org.mule.jms.commons.internal.publish.JmsMessageProducer;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;

import javax.jms.CompletionListener;
import javax.jms.JMSException;
import javax.jms.JMSSecurityException;
import javax.jms.Message;

import org.slf4j.Logger;

/**
 * {@link CompletionListener} for the {@link JmsPublish publish operation}
 *
 * @since 1.0.0
 */
public class JmsPublisherCompletionListener implements CompletionListener {

  private static final Logger LOGGER = getLogger(JmsPublisherCompletionListener.class);
  private static final Result<Void, Void> VOID_RESULT = Result.<Void, Void>builder().build();

  private CompletionCallback<Void, Void> callback;
  private final JmsSession session;
  private final JmsMessageProducer producer;
  private final String destination;
  private final String destinationType;
  private boolean isPartOfCurrentTx;
  private Scheduler scheduler;

  public JmsPublisherCompletionListener(CompletionCallback<Void, Void> callback,
                                        JmsSession session,
                                        JmsMessageProducer producer,
                                        String destination,
                                        String destinationType,
                                        boolean isPartOfCurrentTx,
                                        Scheduler scheduler) {
    this.callback = callback;
    this.session = session;
    this.producer = producer;
    this.destination = destination;
    this.destinationType = destinationType;
    this.isPartOfCurrentTx = isPartOfCurrentTx;
    this.scheduler = scheduler;
  }

  @Override
  public void onCompletion(Message message) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug(String.format("Finished [publish] to the %s: [%s] using session [%s]",
                                 destinationType, destination, session.get()));
    }

    // This is executed asynchronously because is not allowed to be done inside these callbacks
    scheduler.submit(() -> releaseResources(session, isPartOfCurrentTx, producer));
    callback.success(VOID_RESULT);
  }

  @Override
  public void onException(Message message, Exception exception) {
    Exception err = exception;
    if (exception instanceof JMSSecurityException) {
      String msg = format("A security error occurred while sending a message to the %s: [%s]: %s",
                          destinationType, destination, exception.getMessage());
      err = new JmsSecurityException(msg, exception);
    } else if (exception instanceof JmsExtensionException) {
      err = exception;
    } else {
      String msg = format("[publish-onException] An error occurred while sending a message to the %s: [%s]: %s",
                          destinationType, destination, exception.getMessage());
      err = new JmsPublishException(msg, exception);
      if (exception instanceof JMSException) {
        JMSException ex = ((JMSException) exception);
        String errCode = ex.getErrorCode();
        String errMsg =
            format("class: %s, errCode: %s, errMsg:%s, linkedEx: %s", ex.getClass().toString(), errCode, ex.getMessage(),
                   ex.getLinkedException());
        if ((errCode != null) && DETAILED_JMS_EXCEPTION_CODES.contains(errCode)) {
          if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("[publish-onException] Triggering reconnection. errMsg: " + errMsg);
          }
          err = new ConnectionException(msg, exception);
        } else {
          if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("[publish-onException] Not triggering reconnection. errMsg: " + errMsg);
          }
        }
      }
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug(String.format("Failed to [publish] to the %s: [%s] using session [%s]",
                                   destinationType, destination, session.get(), err));
      }
    }
    // This is executed asynchronously because is not allowed to be done inside these callbacks
    scheduler.submit(() -> releaseResources(session, isPartOfCurrentTx, producer));
    callback.error(err);
  }
}
