/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.connection;

import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.slf4j.LoggerFactory.getLogger;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.Destination;
import javax.jms.ExceptionListener;
import javax.jms.JMSContext;
import javax.jms.JMSException;
import javax.jms.MessageProducer;
import javax.jms.QueueConnection;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.jms.TemporaryQueue;
import javax.jms.TemporaryTopic;
import javax.jms.TopicConnection;
import javax.jms.TopicSession;

import org.mule.jms.commons.api.connection.JmsReconnectionManager;
import org.mule.jms.commons.api.connection.caching.CachingConfiguration;
import org.mule.jms.commons.internal.support.JmsSupport;
import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.api.lifecycle.Stoppable;
import org.slf4j.Logger;
import org.springframework.jms.connection.CachingConnectionFactory;
import org.springframework.jms.connection.SessionProxy;
import org.springframework.util.ClassUtils;

/**
 * Decorates the JMS {@link javax.jms.ConnectionFactory} in order to ensure JMS session instances are reused. Applies only if the
 * supplied connection factory isn't already an instance of {@link CachingConnectionFactory} NOTE: Currently only Non-XA JMS
 * {@link javax.jms.ConnectionFactory}'s will be decorated to provide caching.
 *
 * @since 1.0
 */
public class IBMJmsCachingConnectionFactory extends CachingConnectionFactory implements Stoppable, Disposable {

  private static final Logger LOGGER = getLogger(IBMJmsCachingConnectionFactory.class);

  private final String username;
  private final String password;
  private final JmsSupport jmsSupport;
  private final String clientId;
  private final ExceptionListener exceptionListener;
  private JmsReconnectionManager reconnectionManager;

  public IBMJmsCachingConnectionFactory(ConnectionFactory targetConnectionFactory, String username, String password,
                                        String clientId,
                                        CachingConfiguration config, JmsSupport jmsSupport, ExceptionListener exceptionListener,
                                        final JmsReconnectionManager priorityManager) {
    super(targetConnectionFactory);
    checkArgument(!(targetConnectionFactory instanceof CachingConnectionFactory),
                  "The ConnectionFactory provided shouldn't be wrapped in a IBMJmsCachingConnectionFactory");

    super.setCacheConsumers(config.isConsumersCache());
    super.setCacheProducers(config.isProducersCache());
    super.setSessionCacheSize(config.getSessionCacheSize());
    super.setReconnectOnException(true);

    this.exceptionListener = exceptionListener;
    this.username = username;
    this.password = password;
    this.clientId = clientId;
    this.jmsSupport = jmsSupport;
    this.reconnectionManager = priorityManager;
  }

  // MULE-16778: By adding synchronization between the onException and the doCreateConnection method, once Mule reconnection logic
  // starts
  // trying to re-create the connection, is will block until both exceptionListeners handle the exception. That is: the
  // reconnection chain is subscribed to, which triggers Mule's reconnection logic, and the exception is handled by Spring's
  // SingleConnectionFactory and CachingConnectionFactory, which clear the cached sessions, and cleans the underlying single
  // Connection.
  // MULE-18069: Changed synchronization to avoid deadlock: we need that while super.onException is called, doCreateConnection
  // must not get blocked, because it will be holding a lock that will be required by such super.onException method.
  // Therefore, in case of reconnection, onException will be blocked while doCreateConnection is invoked. On the other hand,
  // doCreateConnection will throw a JMSException if reconnection has started. This way, the mentioned lock will be freed
  // and onException can continue its execution.
  @Override
  public void onException(JMSException e) {
    synchronized (this) {
      reconnectionManager.reconnecting();
    }
    // First handle exception on the mule side
    LOGGER.info("Handling exception in Mule's CompositeExceptionListener: {}", e);
    exceptionListener.onException(e);
    // Then handle in Spring's connection factory side
    LOGGER.info("Handling exception in Spring's SingleConnectionFactory: {}", e);
    super.onException(e);
    synchronized (this) {
      reconnectionManager.finishReconnecting();
    }
  }

  public void destroy() {
    if (!reconnectionManager.isReconnecting()) {
      super.destroy();
    }
  }

  protected Connection getConnection() throws JMSException {
    if (reconnectionManager.isReconnecting()) {
      throw new JMSException("Creating a connection while reconnecting");
    }
    return super.getConnection();
  }

  protected Session getSession(final Connection con, final Integer mode) throws JMSException {
    if (reconnectionManager.isReconnecting()) {
      throw new JMSException("Creating a session while reconnecting");
    }
    final Session target = super.getSession(con, mode);
    final List<Class<?>> classes = new ArrayList<Class<?>>(3);
    classes.add(SessionProxy.class);
    if (target instanceof QueueSession) {
      classes.add(QueueSession.class);
    }
    if (target instanceof TopicSession) {
      classes.add(TopicSession.class);
    }
    return (Session) Proxy.newProxyInstance(SessionProxy.class.getClassLoader(), ClassUtils.toClassArray((Collection) classes),
                                            new IBMSessionProxyHandler(target, reconnectionManager, isCacheProducers()));
  }

  @Override
  protected synchronized Connection doCreateConnection() throws JMSException {
    if (reconnectionManager.isReconnecting()) {
      throw new JMSException("Creating a connection while reconnecting");
    }
    Connection connection;
    if (isBlank(username)) {
      connection = jmsSupport.createConnection(getTargetConnectionFactory());
    } else {
      connection = jmsSupport.createConnection(getTargetConnectionFactory(), username, password);
    }

    if (!isBlank(clientId)) {
      connection.setClientID(clientId);
    }

    return connection;
  }

  @Override
  public void stop() {
    resetConnection();
  }

  @Override
  public void dispose() {
    destroy();
  }

  @Override
  public Connection createConnection(String username, String password) throws JMSException {
    throw new javax.jms.IllegalStateException(
                                              "JmsCachingConnectionFactory does not support creating a connection with username and password. Provide the desired username and password when the instance is defined");
  }

  @Override
  public QueueConnection createQueueConnection(String username, String password) throws JMSException {
    throw new javax.jms.IllegalStateException(
                                              "JmsCachingConnectionFactory does not support creating a connection with username and password. Provide the desired username and password when the instance is defined");
  }

  @Override
  public TopicConnection createTopicConnection(String username, String password) throws JMSException {
    throw new javax.jms.IllegalStateException(
                                              "JmsCachingConnectionFactory does not support creating a connection with username and password. Provide the desired username and password when the instance is defined");
  }

  @Override
  public JMSContext createContext() {
    // We'll use the classic API
    return null;
  }

  @Override
  public JMSContext createContext(String userName, String password) {
    // We'll use the classic API
    return null;
  }

  @Override
  public JMSContext createContext(String userName, String password, int sessionMode) {
    // We'll use the classic API
    return null;
  }

  @Override
  public JMSContext createContext(int sessionMode) {
    // We'll use the classic API
    return null;
  }

  private static class IBMSessionProxyHandler implements InvocationHandler {

    Session target;
    private JmsReconnectionManager reconnectionManager;
    private boolean cachedProducers;

    public IBMSessionProxyHandler(final Session target, final JmsReconnectionManager reconnectionManager,
                                  boolean cachedProducers) {
      this.target = target;
      this.reconnectionManager = reconnectionManager;
      this.cachedProducers = cachedProducers;
    }

    @Override
    public Object invoke(final Object proxy, final Method method, final Object[] args) throws Throwable {
      try {
        if (reconnectionManager.isReconnecting()) {
          throw new JMSException("The connection is being reset");
        }

        String methodName = method.getName();

        if (methodName.equals("close")) {
          synchronized (this) {
            return method.invoke(this.target, args);
          }
        }

        if (cachedProducers && (methodName.equals("createProducer") ||
            methodName.equals("createSender") || methodName.equals("createPublisher"))) {
          // Destination argument being null is ok for a producer
          Destination dest = (Destination) args[0];
          if (!(dest instanceof TemporaryQueue || dest instanceof TemporaryTopic)) {
            final List<Class<?>> classes = new ArrayList<Class<?>>(1);
            classes.add(MessageProducer.class);
            return Proxy.newProxyInstance(MessageProducer.class.getClassLoader(), ClassUtils.toClassArray(classes),
                                          new IBMProducerProxyHandler(method.invoke(this.target, args), this));
          }
        }

        return method.invoke(this.target, args);
      } catch (InvocationTargetException ex) {
        throw ex.getTargetException();
      }
    }
  }


  private static class IBMProducerProxyHandler implements InvocationHandler {

    private Object target;
    private IBMSessionProxyHandler ibmSessionProxyHandler;

    public IBMProducerProxyHandler(Object target, IBMSessionProxyHandler ibmSessionProxyHandler) {
      this.target = target;
      this.ibmSessionProxyHandler = ibmSessionProxyHandler;
    }

    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
      try {
        String methodName = method.getName();
        if (methodName.equals("send")) {
          synchronized (ibmSessionProxyHandler) {
            return method.invoke(this.target, args);
          }
        }
        return method.invoke(this.target, args);
      } catch (InvocationTargetException ex) {
        throw ex.getTargetException();
      }
    }

  }
}
