/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.api.message;

import static org.mule.runtime.api.util.Preconditions.checkArgument;

import java.io.Serializable;

public class AbstractJmsAttributes implements Serializable {

  private static final long serialVersionUID = -8148917084189760450L;

  /**
   * Container element for all the properties present in a JMS Message
   */
  private final JmsMessageProperties properties;

  /**
   * All the possible headers of a JMS Message
   */
  private final JmsHeaders headers;

  /**
   * The session ACK ID required to ACK a the current Message if one is available, or null otherwise.
   */
  private final String ackId;

  public AbstractJmsAttributes(JmsMessageProperties properties, JmsHeaders headers, String ackId) {
    this.properties = properties;
    this.headers = headers;
    this.ackId = ackId;
  }

  /**
   * {@inheritDoc}
   */
  public JmsMessageProperties getProperties() {
    return properties;
  }

  /**
   * {@inheritDoc}
   */
  public JmsHeaders getHeaders() {
    return headers;
  }

  /**
   * {@inheritDoc}
   */
  public String getAckId() {
    return ackId;
  }

  /**
   * Builder implementation for creating a {@link DefaultJmsAttributes} instance
   * 
   * @since 1.0
   */
  public static class Builder {

    private JmsMessageProperties properties;
    private JmsHeaders headers;
    private String ackId;

    private Builder() {}

    public static Builder newInstance() {
      return new Builder();
    }

    public Builder withProperties(JmsMessageProperties properties) {
      this.properties = properties;
      return this;
    }

    public Builder withHeaders(JmsHeaders headers) {
      this.headers = headers;
      return this;
    }

    public Builder withAckId(String ackId) {
      this.ackId = ackId;
      return this;
    }

    public JmsAttributes build() {
      checkArgument(properties != null, "No JmsMessageProperties were provided, but they are required for the JmsAttributes");
      checkArgument(headers != null, "No JmsHeaders were provided, but they are required for the JmsAttributes");
      return new DefaultJmsAttributes(properties, headers, ackId);
    }
  }

}
