/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.connection.provider;

import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.core.api.MuleContext;

import com.mulesoft.mule.runtime.bti.api.jms.ConnectionFactoryDecorator;
import com.mulesoft.mule.runtime.bti.api.jms.JmsConnectionConfig;
import com.mulesoft.mule.runtime.bti.api.jms.JmsConnectionConfig.Builder;

import java.util.Collection;

import javax.jms.ConnectionFactory;
import javax.jms.ExceptionListener;

/**
 * Decorator for {@link ConnectionFactory} to intercept method calls and apply logic based on the {@link ConnectionFactory} type and
 * the transactional context in which is being used.
 *
 * @since 1.4
 */
public class InternalConnectionFactoryDecorator {

  private MuleContext muleContext;
  private Registry registry;

  public InternalConnectionFactoryDecorator(MuleContext muleContext, Registry registry) {
    this.muleContext = muleContext;
    this.registry = registry;
  }

  public ConnectionFactory decorate(ConnectionFactory connectionFactory, String configName, boolean reuseSessions,
                                    String username, String password, String clientId, ExceptionListener exceptionListener) {
    Collection<ConnectionFactoryDecorator> decorators = registry.lookupAllByType(ConnectionFactoryDecorator.class);
    ConnectionFactory decoratedConnectionFactory = connectionFactory;
    if (!decorators.isEmpty()) {
      JmsConnectionConfig build = new Builder(configName)
          .setReuseSessions(reuseSessions)
          .setUserName(username)
          .setPassword(password)
          .setClientId(clientId)
          .setExceptionListener(exceptionListener)
          .build();
      for (ConnectionFactoryDecorator decorator : decorators) {
        if (decorator.appliesTo(decoratedConnectionFactory, muleContext)) {
          decoratedConnectionFactory = decorator.decorate(decoratedConnectionFactory, build, muleContext);
        }
      }
    }

    return decoratedConnectionFactory;
  }
}
