/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.source;

import static org.mule.jms.commons.internal.common.JmsCommons.resolveMessageContentType;
import static org.mule.jms.commons.internal.common.JmsCommons.resolveMessageEncoding;
import static org.mule.jms.commons.internal.common.JmsCommons.resolveOverride;
import static org.mule.jms.commons.internal.config.InternalAckMode.IMMEDIATE;
import static org.mule.jms.commons.internal.config.InternalAckMode.MANUAL;
import static org.slf4j.LoggerFactory.getLogger;
import org.mule.jms.commons.api.connection.JmsSpecification;
import org.mule.jms.commons.api.message.JmsAttributes;
import org.mule.jms.commons.internal.config.InternalAckMode;
import org.mule.jms.commons.internal.config.JmsConfig;
import org.mule.jms.commons.internal.connection.session.JmsSession;
import org.mule.jms.commons.internal.connection.session.JmsSessionManager;
import org.mule.jms.commons.internal.message.JmsResultFactory;
import org.mule.jms.commons.internal.source.push.JmsMessageListener;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.source.SourceCallback;
import org.mule.runtime.extension.api.runtime.source.SourceCallbackContext;

import java.util.function.Supplier;

import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Message;

import org.slf4j.Logger;

/**
 * Common utility which encapsulates the logic of dispatching a received message to a flow.
 *
 * @since 1.0.0
 */
public class JmsMessageDispatcher {

  private static final Logger LOGGER = getLogger(JmsMessageListener.class);

  private final String defaultEncoding;
  private final String defaultContentType;
  private final JmsSessionManager sessionManager;
  private final SourceCallback sourceCallback;
  private final JmsConnectionExceptionResolver exceptionResolver;

  private JmsConfig config;
  private JmsSpecification specification;
  private Supplier<JmsSession> session;
  private InternalAckMode ackMode;
  private JmsListenerLock jmsListenerLock;
  private JmsResultFactory resultFactory = JmsResultFactory.getInstance();

  public JmsMessageDispatcher(JmsConfig config,
                              String defaultContentType,
                              String defaultEncoding,
                              JmsSpecification specification,
                              Supplier<JmsSession> session,
                              InternalAckMode ackMode,
                              JmsSessionManager sessionManager,
                              SourceCallback sourceCallback,
                              JmsListenerLock jmsListenerLock,
                              JmsConnectionExceptionResolver exceptionResolver) {
    this.config = config;
    this.defaultContentType = defaultContentType;
    this.defaultEncoding = defaultEncoding;
    this.specification = specification;
    this.session = session;
    this.ackMode = ackMode;
    this.sessionManager = sessionManager;
    this.sourceCallback = sourceCallback;
    this.jmsListenerLock = jmsListenerLock;
    this.exceptionResolver = exceptionResolver;
  }

  /**
   * Dispatches the given message to the current source callback.
   *
   * @param message Message to dispatch
   * @param context Context to enrich with message and session information.
   */
  public void dispatchMessage(Message message, SourceCallbackContext context) throws Exception {
    try {
      saveReplyToDestination(message, context);
      context.setCorrelationId(getCorrelationId(message));

      Result<Object, JmsAttributes> result = resultFactory.createResult(message, specification, resolveContentType(message),
                                                                        resolveEncoding(message), session.get().getAckId());

      registerAckIdIfRequired(message);

      sourceCallback.handle(result, context);

      doAckIfRequired(message);

    } catch (Exception e) {
      String msg = String.format("An error occurred while dispatching a Message from the listener on session [%s]: %s",
                                 session.get(), e.getMessage());
      LOGGER.error(msg, e);

      JmsListener.notifyIfConnectionProblem(sourceCallback, e, exceptionResolver);

      throw e;
    }
  }

  private String resolveContentType(Message message) {
    // If no explicit content type was provided to the operation, fallback to the
    // one communicated in the message properties. Finally if no property was set,
    // use the default one provided by the config
    return resolveOverride(resolveMessageContentType(message, config.getContentType()), defaultContentType);
  }

  private String resolveEncoding(Message message) {
    // If no explicit content type was provided to the operation, fallback to the
    // one communicated in the message properties. Finally if no property was set,
    // use the default one provided by the config
    return resolveOverride(resolveMessageEncoding(message, config.getEncoding()), defaultEncoding);
  }

  private void saveReplyToDestination(Message message, SourceCallbackContext context) {
    try {
      Destination replyTo = message.getJMSReplyTo();
      if (replyTo != null) {
        context.addVariable(JmsListener.REPLY_TO_DESTINATION_VAR, replyTo);
      }
    } catch (JMSException e) {
      LOGGER.error("An error occurred while obtaining the ReplyTo destination: ", e);
      JmsListener.notifyIfConnectionProblem(sourceCallback, e, exceptionResolver);
    }
  }

  private void registerAckIdIfRequired(Message message) {
    if (ackMode.equals(MANUAL)) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Registering pending ACK on session: " + session.get().getAckId());
      }
      String id = session.get().getAckId()
          .orElseThrow(() -> new IllegalArgumentException("An AckId is required when MANUAL AckMode is set"));

      sessionManager.registerMessageForAck(id, message, session.get().get(), jmsListenerLock);
    }
  }

  private void doAckIfRequired(Message message) throws JMSException {
    if (ackMode.equals(IMMEDIATE)) {
      LOGGER.debug("Automatically performing an ACK over the message, since AckMode was IMMEDIATE");
      message.acknowledge();
    }
  }

  private String getCorrelationId(Message message) {
    try {
      return message.getJMSCorrelationID();
    } catch (JMSException e) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Failed to obtain correlationId from message", e);
      }

      return null;
    }
  }
}
