/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.connection.provider;

import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;

import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.core.api.MuleContext;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collection;

import javax.jms.ConnectionFactory;
import javax.jms.ExceptionListener;

/**
 * Decorator for {@link ConnectionFactory} to intercept method calls and apply logic based on the {@link ConnectionFactory} type and
 * the transactional context in which is being used.
 *
 * @since 1.4
 */
public class InternalConnectionFactoryDecorator implements ConnectionFactoryDecorator {

  private MuleContext muleContext;
  private Registry registry;
  private final Constructor<?> constructor;

  private final Method setReuseSessions;
  private final Method setUserName;
  private final Method setPassword;
  private final Method setClientId;
  private final Method setExceptionListener;
  private final Method buildMethod;

  private final Method appliesTo;
  private final Method decorate;

  private final LazyValue<Collection> factoryDecorators;

  InternalConnectionFactoryDecorator(MuleContext muleContext, Registry registry) {
    this.muleContext = muleContext;
    this.registry = registry;

    try {
      Class<?> decoratorClass = Class.forName("com.mulesoft.mule.runtime.bti.api.jms.ConnectionFactoryDecorator");

      Class<?> decoratorBuilderClass = Class.forName("com.mulesoft.mule.runtime.bti.api.jms.JmsConnectionConfig$Builder");
      Class jmsConfigClass = Class.forName("com.mulesoft.mule.runtime.bti.api.jms.JmsConnectionConfig");

      constructor = decoratorBuilderClass.getConstructor(String.class);

      setReuseSessions = decoratorBuilderClass.getMethod("setReuseSessions", boolean.class);
      setUserName = decoratorBuilderClass.getMethod("setUserName", String.class);
      setPassword = decoratorBuilderClass.getMethod("setPassword", String.class);
      setClientId = decoratorBuilderClass.getMethod("setClientId", String.class);
      setExceptionListener = decoratorBuilderClass.getMethod("setExceptionListener", ExceptionListener.class);
      buildMethod = decoratorBuilderClass.getMethod("build");

      appliesTo = decoratorClass.getMethod("appliesTo", ConnectionFactory.class, MuleContext.class);
      decorate = decoratorClass.getMethod("decorate", ConnectionFactory.class, jmsConfigClass, MuleContext.class);

      factoryDecorators = new LazyValue<>(() -> {
        try {
          return registry.lookupAllByType(decoratorClass);
        } catch (Exception e) {
          throw new RuntimeException(e);
        }
      });
    } catch (ClassNotFoundException | NoSuchMethodException e) {
      throw new MuleRuntimeException(createStaticMessage("An error occurred trying to load XA Connection Factory Decorator"), e);
    }
  }

  /**
   * Decorates a Connection Factory with the available Connection Factory Decorators on the Mule Runtime.
   *
   * Dev Notes: To avoid the JMS Client require on Bitronix Module to compile, this class uses reflection over the module.
   *
   * @param connectionFactory Connection Factory to decorate
   * @param configName Global Element name of the JMS Configuration
   * @param reuseSessions If the sessions should be reused
   * @param username Username of the JMS Connection
   * @param password Password of the JMS Connection
   * @param clientId ClientID of the JMS Connection
   * @param exceptionListener ExceptionListener of the JMS Connection
   *
   * @return A possible decorated ConnectionFactory, the same will be returned in case of no decorator found.
   */
  public ConnectionFactory decorate(ConnectionFactory connectionFactory, String configName, boolean reuseSessions,
                                    String username, String password, String clientId, ExceptionListener exceptionListener) {


    ConnectionFactory decoratedConnectionFactory;
    try {

      Collection<?> decorators = this.factoryDecorators.get();
      decoratedConnectionFactory = connectionFactory;

      if (!decorators.isEmpty()) {
        Object jmsConfigBuilder = constructor.newInstance(configName);
        setReuseSessions.invoke(jmsConfigBuilder, reuseSessions);
        setUserName.invoke(jmsConfigBuilder, username);
        setPassword.invoke(jmsConfigBuilder, password);
        setClientId.invoke(jmsConfigBuilder, clientId);
        setExceptionListener.invoke(jmsConfigBuilder, exceptionListener);

        Object jmsConfig = buildMethod.invoke(jmsConfigBuilder);

        for (Object decorator : decorators) {
          if (appliesTo(appliesTo, decoratedConnectionFactory, decorator)) {
            decoratedConnectionFactory = decorate(decorate, decoratedConnectionFactory, jmsConfig, muleContext, decorator);
          }
        }

      }
    } catch (InstantiationException | IllegalAccessException
        | InvocationTargetException e) {
      throw new RuntimeException(e);
    }

    return decoratedConnectionFactory;
  }

  private ConnectionFactory decorate(Method decorate, ConnectionFactory decoratedConnectionFactory, Object build,
                                     MuleContext muleContext, Object decorator) {
    try {
      return (ConnectionFactory) decorate.invoke(decorator, decoratedConnectionFactory, build, muleContext);
    } catch (IllegalAccessException | InvocationTargetException e) {
      throw new MuleRuntimeException(createStaticMessage("An error occurred trying to decorate a Connection Factory."), e);
    }
  }

  private boolean appliesTo(Method appliesTo, ConnectionFactory decoratedConnectionFactory, Object decorator) {
    try {
      return (boolean) appliesTo.invoke(decorator, decoratedConnectionFactory, muleContext);
    } catch (IllegalAccessException | InvocationTargetException e) {
      throw new MuleRuntimeException(createStaticMessage("An error occurred trying to check if the Connection Factory can be decorated."),
                                     e);
    }
  }
}
