/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.api.message;

import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.jms.commons.api.config.DefaultJmsProducerConfig;
import org.mule.jms.commons.api.destination.JmsDestination;
import org.mule.jms.commons.api.exception.DestinationNotFoundException;
import org.mule.jms.commons.internal.common.JmsCommons;
import org.mule.jms.commons.internal.config.JmsConfig;
import org.mule.jms.commons.internal.message.JMSXDefinedPropertiesNames;
import org.mule.jms.commons.internal.message.JmsMessageUtils;
import org.mule.jms.commons.internal.support.JmsSupport;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.runtime.parameter.CorrelationInfo;
import org.mule.runtime.extension.api.runtime.parameter.OutboundCorrelationStrategy;

import java.nio.charset.Charset;
import java.util.Map;

import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.Session;

import org.slf4j.Logger;

public class JmsMessageFactory {


  private static final Logger LOGGER = getLogger(JmsMessageFactory.class);
  public static final String BODY_CONTENT_TYPE_JMS_PROPERTY = "MM_MESSAGE_CONTENT_TYPE";
  public static final String BODY_ENCODING_JMS_PROPERTY = "MM_MESSAGE_ENCODING";

  /**
   * Creates a {@link Message} based on the provided configurations
   *
   * @param jmsSupport                  the {@link JmsSupport} used to create the JMSReplyTo {@link Destination}
   * @param outboundCorrelationStrategy the correlationId handling strategy
   * @param correlationInfo             the correlation information for the current message
   * @param session                     the current {@link Session}
   * @param config                      the current {@link DefaultJmsProducerConfig}
   * @return the {@link Message} created by the user
   * @throws JMSException if an error occurs
   */
  public static Message build(JmsSupport jmsSupport, OutboundCorrelationStrategy outboundCorrelationStrategy,
                              CorrelationInfo correlationInfo,
                              Session session, JmsConfig config,
                              JmsMessageBuilder messageBuilder)
      throws JMSException {

    TypedValue<Object> body = messageBuilder.getBody();
    Message message = JmsMessageUtils.toMessage(body, session);

    setJmsCorrelationIdHeader(message, outboundCorrelationStrategy, correlationInfo, messageBuilder.getCorrelationId());
    setJmsTypeHeader(message, messageBuilder.getJmsType());
    setJmsReplyToHeader(jmsSupport, session, message, messageBuilder.getReplyTo(), config);

    setJmsxProperties(message, messageBuilder.getJmsxProperties());
    setUserProperties(message, messageBuilder.getProperties());

    if (messageBuilder.isSendContentType()) {
      setContentTypeProperty(message, body.getDataType(), messageBuilder.getOutboundContentType());
    }
    if (messageBuilder.isSendEncoding()) {
      setEncodingProperty(message, body.getDataType(),
                          JmsCommons.resolveOverride(config.getEncoding(), messageBuilder.getOutboundEncoding()));
    }

    return message;
  }

  private static void setJmsReplyToHeader(JmsSupport jmsSupport, Session session, Message message,
                                          JmsDestination replyDestination, JmsConfig config) {
    try {
      if (replyDestination != null &&
          !isBlank(replyDestination.getDestination())) {
        Destination destination = jmsSupport.createDestination(session, replyDestination.getDestination(),
                                                               replyDestination.getDestinationType().isTopic(), config);
        message.setJMSReplyTo(destination);
      }
    } catch (DestinationNotFoundException | JMSException e) {
      LOGGER.error("Unable to set JMSReplyTo header: ", e);
    }
  }

  private static void setEncodingProperty(Message message, DataType dataType, String defaultEncoding) {
    try {
      message.setStringProperty(BODY_ENCODING_JMS_PROPERTY,
                                dataType.getMediaType().getCharset().map(Charset::toString).orElse(defaultEncoding));
    } catch (JMSException e) {
      LOGGER.error(format("Unable to set property [%s] of type String: ", BODY_ENCODING_JMS_PROPERTY), e);
    }
  }

  private static void setContentTypeProperty(Message message, DataType dataType, String outboundContentType) {
    try {
      String value = isBlank(outboundContentType) ? dataType.getMediaType().toRfcString() : outboundContentType;
      message.setStringProperty(BODY_CONTENT_TYPE_JMS_PROPERTY, value);
    } catch (JMSException e) {
      LOGGER.error(format("Unable to set property [%s] of type String: ", BODY_CONTENT_TYPE_JMS_PROPERTY), e);
    }
  }

  private static void setJmsxProperties(final Message message, JmsxProperties jmsxProperties) {
    jmsxProperties.asMap().entrySet().stream()
        .filter(e -> e.getValue() != null)
        .forEach(e -> setJmsPropertySanitizeKeyIfNecessary(message, e.getKey(), e.getValue()));
  }

  private static void setUserProperties(final Message message, Map<String, Object> properties) {
    properties.keySet().stream()
        .filter(key -> !isBlank(key) && !JMSXDefinedPropertiesNames.JMSX_NAMES.contains(key))
        .forEach(key -> setJmsPropertySanitizeKeyIfNecessary(message, key, properties.get(key)));
  }

  private static void setJmsPropertySanitizeKeyIfNecessary(Message msg, String key, Object value) {
    try {
      // sanitize key as JMS Property Name
      key = JmsMessageUtils.encodeKey(key);
      if (value instanceof TypedValue) {
        value = ((TypedValue) value).getValue();
      }
      msg.setObjectProperty(key, value);
    } catch (JMSException e) {
      // Various JMS servers have slightly different rules to what
      // can be set as an object property on the message; therefore
      // we have to take a hit n' hope approach
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug(format("Unable to set property [%s] of type [%s]: ", key, value.getClass().getSimpleName()), e);
      }
    }
  }

  private static void setJmsTypeHeader(Message message, String jmsType) {
    try {
      if (!isBlank(jmsType)) {
        message.setJMSType(jmsType);
      }
    } catch (JMSException e) {
      LOGGER.error("An error occurred while setting the JMSType property: %s", e);
    }
  }

  private static void setJmsCorrelationIdHeader(Message message,
                                                OutboundCorrelationStrategy outboundCorrelationStrategy,
                                                CorrelationInfo correlationInfo, String correlationId) {
    outboundCorrelationStrategy.getOutboundCorrelationId(correlationInfo, correlationId).ifPresent(id -> {
      try {
        message.setJMSCorrelationID(id);
      } catch (JMSException e) {
        LOGGER.error("An error occurred while setting the JMSCorrelationID property: %s", e);
      }
    });
  }
}
