/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.operation;

import static java.lang.String.format;
import static org.mule.jms.commons.internal.common.JmsCommons.QUEUE;
import static org.mule.jms.commons.internal.common.JmsCommons.createJmsSession;
import static org.mule.jms.commons.internal.common.JmsCommons.getDestinationType;
import static org.mule.jms.commons.internal.common.JmsCommons.isPartOfCurrentTx;
import static org.mule.jms.commons.internal.common.JmsCommons.releaseResources;
import static org.mule.jms.commons.internal.config.InternalAckMode.AUTO;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.jms.commons.api.config.DefaultJmsProducerConfig;
import org.mule.jms.commons.api.destination.DestinationType;
import org.mule.jms.commons.api.destination.DestinationTypeDescriptor;
import org.mule.jms.commons.api.exception.JmsExtensionException;
import org.mule.jms.commons.api.exception.JmsPublishException;
import org.mule.jms.commons.api.exception.JmsPublisherErrorTypeProvider;
import org.mule.jms.commons.api.exception.JmsSecurityException;
import org.mule.jms.commons.api.message.DefaultJmsMessageBuilder;
import org.mule.jms.commons.api.message.JmsMessageBuilder;
import org.mule.jms.commons.internal.common.JmsPublisherCompletionListener;
import org.mule.jms.commons.internal.config.JmsConfig;
import org.mule.jms.commons.internal.connection.JmsConnection;
import org.mule.jms.commons.internal.connection.JmsTransactionalConnection;
import org.mule.jms.commons.internal.connection.session.JmsSession;
import org.mule.jms.commons.internal.connection.session.JmsSessionManager;
import org.mule.jms.commons.internal.publish.JmsMessageProducer;
import org.mule.jms.commons.internal.publish.PublisherParameters;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.ConfigOverride;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.ParameterGroup;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.parameter.CorrelationInfo;
import org.mule.runtime.extension.api.runtime.parameter.OutboundCorrelationStrategy;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;
import org.mule.runtime.extension.api.tx.OperationTransactionalAction;

import javax.jms.Destination;
import javax.jms.IllegalStateException;
import javax.jms.JMSException;
import javax.jms.JMSSecurityException;
import javax.jms.Message;

import org.slf4j.Logger;

/**
 * Operation that allows the user to send a message to a JMS {@link Destination}
 *
 * @since 1.0
 */
public final class JmsPublish implements Disposable {

  private static final Logger LOGGER = getLogger(JmsPublish.class);
  private static final Result<Void, Void> VOID_RESULT = Result.<Void, Void>builder().build();


  public JmsPublish(JmsSessionManager jmsSessionManager, SchedulerService schedulerService) {
    this.jmsSessionManager = jmsSessionManager;
    this.scheduler = schedulerService.ioScheduler();
  }

  private JmsSessionManager jmsSessionManager;

  private Scheduler scheduler;

  /**
   * Operation that allows the user to send a {@link Message} to a JMS {@link Destination}
   *
   * @param config              the current {@link DefaultJmsProducerConfig }
   * @param connection          the current {@link JmsConnection}
   * @param destination         the name of the {@link Destination} where the {@link Message} should be sent
   * @param destinationType     the {@link DestinationType} of the {@code destination}
   * @param messageBuilder      the {@link DefaultJmsMessageBuilder } used to create the {@link Message} to be sent
   * @param overrides           Parameter Group with overriding parameters from the configuration
   * @param transactionalAction Transactional Action for the operation. Indicates if the publish must be executed
   *                            or not in a transaction.
   * @param sendCorrelationId   options on whether to include an outbound correlation id or not
   * @param correlationInfo     the current message's correlation info
   * @throws JmsExtensionException if an error occurs trying to publish a message
   */
  @Throws(JmsPublisherErrorTypeProvider.class)
  public void publish(@Config JmsConfig config, @Connection JmsTransactionalConnection connection,
                      @Summary("The name of the Destination where the Message should be sent") String destination,
                      @Optional(
                          defaultValue = QUEUE) @Summary("The type of the Destination") DestinationTypeDescriptor destinationType,
                      @Summary("A builder for the message that will be published") @ParameterGroup(name = "Message",
                          showInDsl = true) JmsMessageBuilder messageBuilder,
                      @ParameterGroup(name = "Publish Configuration") PublisherParameters overrides,
                      OperationTransactionalAction transactionalAction,
                      @ConfigOverride OutboundCorrelationStrategy sendCorrelationId,
                      CorrelationInfo correlationInfo,
                      CompletionCallback<Void, Void> completionCallback)

      throws JmsExtensionException {

    JmsSession session = null;
    JmsMessageProducer producer = null;
    boolean isPartOfCurrentTx = false;
    try {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Begin [publish] on " + getDestinationType(destinationType) + ": ["
            + destination + "]");
      }

      session = createJmsSession(connection, AUTO, destinationType.isTopic(), jmsSessionManager, transactionalAction);
      isPartOfCurrentTx = isPartOfCurrentTx(session, connection, jmsSessionManager);

      Message message = messageBuilder.build(connection.getJmsSupport(),
                                             sendCorrelationId,
                                             correlationInfo,
                                             session.get(),
                                             config);

      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug(format("Message built, sending message to the %s: [%s] using session [%s]",
                            getDestinationType(destinationType), destination, session.get()));
      }

      Destination jmsDestination = connection.getJmsSupport()
          .createDestination(session.get(), destination, destinationType.isTopic(), config);

      producer = connection.createProducer(session, jmsDestination, destinationType.isTopic());
      if (isPartOfCurrentTx) {
        publishTransactionally(session, producer, message, overrides, completionCallback);
      } else {
        producer.publish(message, overrides,
                         new JmsPublisherCompletionListener(completionCallback, session, producer,
                                                            destination,
                                                            getDestinationType(destinationType), false, scheduler));
      }

    } catch (JMSSecurityException e) {
      String msg = format("A security error occurred while sending a message to the %s: [%s]: %s",
                          getDestinationType(destinationType), destination, e.getMessage());
      completionCallback.error(new JmsSecurityException(msg, e));
      releaseResources(session, isPartOfCurrentTx, producer);
    } catch (IllegalStateException e) {
      boolean isConnectionError = session == null;
      String msg = format((isConnectionError ? "A connection error" : "An error")
          + " occurred while sending a message to the %s: [%s]: %s",
                          getDestinationType(destinationType), destination, e.getMessage());
      completionCallback.error(isConnectionError ? new ConnectionException(e) : new JmsPublishException(msg, e));
      releaseResources(session, isPartOfCurrentTx, producer);
    } catch (JmsPublishException e) {
      completionCallback.error(e);
      releaseResources(session, isPartOfCurrentTx, producer);
    } catch (Throwable e) {
      String msg = format("An error occurred while sending a message to the %s: [%s]: %s",
                          getDestinationType(destinationType), destination, e.getMessage());
      completionCallback.error(new JmsPublishException(msg, e));
      releaseResources(session, isPartOfCurrentTx, producer);
    }
  }

  /**
   * When inside of a transaction, the same thread that creates the producer should be the same thread the closes it,
   * so using the non blocking API for TXs doesn't makes sense, when the publish is inside the transaction this
   * method should be called.
   */
  private void publishTransactionally(JmsSession session, JmsMessageProducer producer, Message message,
                                      PublisherParameters overrides, CompletionCallback<Void, Void> completionCallback)
      throws JMSException {
    producer.publish(message, overrides);
    completionCallback.success(VOID_RESULT);
    releaseResources(session, true, producer);
  }

  @Override
  public void dispose() {
    if (scheduler != null) {
      scheduler.stop();
    }
  }
}
