/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.source;

import static java.lang.String.format;
import static java.util.Optional.empty;
import static org.mule.jms.commons.api.connection.JmsSpecification.JMS_2_0;
import static org.mule.jms.commons.internal.common.JmsCommons.QUEUE;
import static org.mule.jms.commons.internal.common.JmsCommons.TOPIC;
import static org.mule.jms.commons.internal.common.JmsCommons.getDestinationType;
import static org.mule.jms.commons.internal.common.JmsCommons.isPartOfCurrentTx;
import static org.mule.jms.commons.internal.common.JmsCommons.releaseResources;
import static org.mule.jms.commons.internal.common.JmsCommons.resolveOverride;
import static org.mule.jms.commons.internal.common.JmsCommons.toInternalAckMode;
import static org.mule.jms.commons.internal.config.InternalAckMode.AUTO;
import static org.mule.jms.commons.internal.config.InternalAckMode.TRANSACTED;
import static org.mule.runtime.extension.api.tx.SourceTransactionalAction.ALWAYS_BEGIN;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.jms.commons.api.RequestReplyPattern;
import org.mule.jms.commons.api.config.JmsConsumerConfig;
import org.mule.jms.commons.api.destination.ConsumerType;
import org.mule.jms.commons.api.destination.TopicConsumer;
import org.mule.jms.commons.api.lock.JmsListenerLockFactory;
import org.mule.jms.commons.internal.config.InternalAckMode;
import org.mule.jms.commons.internal.config.JmsAckMode;
import org.mule.jms.commons.internal.config.JmsConfig;
import org.mule.jms.commons.internal.connection.JmsConnection;
import org.mule.jms.commons.internal.connection.JmsTransactionalConnection;
import org.mule.jms.commons.internal.connection.XaJmsTransactionalConnection;
import org.mule.jms.commons.internal.connection.session.JmsSession;
import org.mule.jms.commons.internal.connection.session.JmsSessionManager;
import org.mule.jms.commons.internal.publish.JmsMessageProducer;
import org.mule.jms.commons.internal.source.polling.JmsXaPollingMessageConsumerDelegate;
import org.mule.jms.commons.internal.source.push.JmsMessageListenerDelegate;
import org.mule.jms.commons.internal.source.push.JmsMessageListenerFactory;
import org.mule.jms.commons.internal.support.Jms102bSupport;
import org.mule.jms.commons.internal.support.JmsSupport;
import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionProvider;
import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.message.Error;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.scheduler.SchedulerConfig;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.api.tx.TransactionException;
import org.mule.runtime.core.api.util.StringMessageUtils;
import org.mule.runtime.extension.api.connectivity.XATransactionalConnection;
import org.mule.runtime.extension.api.runtime.parameter.CorrelationInfo;
import org.mule.runtime.extension.api.runtime.source.SourceCallback;
import org.mule.runtime.extension.api.runtime.source.SourceCallbackContext;

import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.Queue;
import javax.jms.Topic;

import org.slf4j.Logger;

import java.util.Optional;

/**
 * JMS Subscriber for {@link Destination}s, allows to listen for incoming {@link Message}s
 *
 * @since 1.0
 */
public class JmsListener {

  private static final Logger LOGGER = getLogger(JmsListener.class);

  static final String REPLY_TO_DESTINATION_VAR = "REPLY_TO_DESTINATION";
  static final String CORRELATION_ID_VAR = "CORRELATION_ID";
  static final String MESSAGE_ID_VAR = "MESSAGE_ID";

  private static final String THREAD_NAME = "JMS-CLIENT-LISTENER";
  private JmsConnectionExceptionResolver exceptionResolver;
  private Scheduler scheduler;
  private JmsSessionManager sessionManager;
  private JmsConfig config;
  private ConnectionProvider<JmsTransactionalConnection> connectionProvider;
  private JmsTransactionalConnection connection;
  private JmsSupport jmsSupport;
  private String destination;
  private ConsumerType consumerType;
  private JmsAckMode ackMode;
  private String selector;
  private String inboundContentType;
  private String inboundEncoding;
  private int numberOfConsumers;
  private MessageConsumerDelegate messageConsumerDelegate;
  private SourceConfiguration sourceConfiguration;
  private SchedulerService schedulerService;
  private JmsResourceReleaser resourceCleaner;
  private final JmsListenerLockFactory lockFactory;
  private Optional<Integer> jmsMaxIdleConnectionTimeout = empty();

  private SourceCallback sourceCallback;

  static void notifyIfConnectionProblem(SourceCallbackContext callbackContext, Exception e,
                                        JmsConnectionExceptionResolver exceptionResolver) {
    notifyIfConnectionProblem(callbackContext.getSourceCallback(), e, exceptionResolver);
  }

  public static void notifyIfConnectionProblem(SourceCallback callback, Exception e,
                                               JmsConnectionExceptionResolver exceptionResolver) {
    exceptionResolver.resolveException(e).ifPresent(ce -> callback.onConnectionException(ce));
  }

  public JmsListener(JmsSessionManager sessionManager,
                     JmsConfig config,
                     ConnectionProvider<JmsTransactionalConnection> connectionProvider,
                     String destination,
                     ConsumerType consumerType,
                     JmsAckMode ackMode,
                     String selector,
                     String inboundContentType,
                     String inboundEncoding,
                     int numberOfConsumers,
                     SourceConfiguration sourceConfiguration,
                     SchedulerService schedulerService,
                     JmsConnectionExceptionResolver exceptionResolver,
                     JmsResourceReleaser resourceCleaner) {
    this(sessionManager, config, connectionProvider, destination, consumerType, ackMode, selector, inboundContentType,
         inboundEncoding, numberOfConsumers, sourceConfiguration, schedulerService, exceptionResolver, resourceCleaner,
         JmsListenerLockFactory.newDefault(), empty());
  }

  public JmsListener(JmsSessionManager sessionManager,
                     JmsConfig config,
                     ConnectionProvider<JmsTransactionalConnection> connectionProvider,
                     String destination,
                     ConsumerType consumerType,
                     JmsAckMode ackMode,
                     String selector,
                     String inboundContentType,
                     String inboundEncoding,
                     int numberOfConsumers,
                     SourceConfiguration sourceConfiguration,
                     SchedulerService schedulerService,
                     JmsConnectionExceptionResolver exceptionResolver,
                     JmsResourceReleaser resourceCleaner,
                     JmsListenerLockFactory lockFactory,
                     Optional<Integer> jmsMaxIdleConnectionTimeout) {
    this.sessionManager = sessionManager;
    this.config = config;
    this.connectionProvider = connectionProvider;
    this.destination = destination;
    this.consumerType = consumerType;
    this.ackMode = ackMode;
    this.selector = selector;
    this.inboundContentType = inboundContentType;
    this.inboundEncoding = inboundEncoding;
    this.numberOfConsumers = numberOfConsumers;
    this.sourceConfiguration = sourceConfiguration;
    this.schedulerService = schedulerService;
    this.exceptionResolver = exceptionResolver;
    this.resourceCleaner = resourceCleaner;
    this.lockFactory = lockFactory;
    this.jmsMaxIdleConnectionTimeout = jmsMaxIdleConnectionTimeout;
  }

  public JmsListener(JmsSessionManager sessionManager,
                     JmsConfig config,
                     ConnectionProvider<JmsTransactionalConnection> connectionProvider,
                     String destination,
                     ConsumerType consumerType,
                     JmsAckMode ackMode,
                     String selector,
                     String inboundContentType,
                     String inboundEncoding,
                     int numberOfConsumers,
                     SourceConfiguration sourceConfiguration,
                     SchedulerService schedulerService) {
    this(sessionManager,
         config,
         connectionProvider,
         destination,
         consumerType,
         ackMode,
         selector,
         inboundContentType,
         inboundEncoding,
         numberOfConsumers,
         sourceConfiguration,
         schedulerService,
         new DefaultJmsConnectionExceptionResolver(),
         new DefaultJmsResourceReleaser());
  }

  public void onStart(SourceCallback sourceCallback) throws MuleException {

    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Starting JMS Message Listener");
    }

    this.sourceCallback = sourceCallback;
    scheduler = schedulerService.ioScheduler(SchedulerConfig.config().withName(THREAD_NAME));

    JmsConsumerConfig consumerConfig = config.getConsumerConfig();

    InternalAckMode resolvedAckMode = sourceConfiguration.getTransactionalAction().equals(ALWAYS_BEGIN)
        ? TRANSACTED
        : resolveOverride(toInternalAckMode(consumerConfig.getAckMode()), toInternalAckMode(ackMode));

    connection = connectionProvider.connect();

    ConnectionValidationResult connectionValidationResult = connectionProvider.validate(connection);

    if (!connectionValidationResult.isValid()) {
      throw new ConnectionException(connectionValidationResult.getException(), connection);
    }

    validateTransactionType(connection);

    jmsSupport = connection.getJmsSupport();

    connection.registerExceptionListener(e -> sourceCallback.onConnectionException(new ConnectionException(e, connection)));

    validateNumberOfConsumers(numberOfConsumers);

    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug(format("Starting JMS Listener with [%s] consumers on destination [%s] of type [%s] with AckMode [%s]",
                          numberOfConsumers, destination, getDestinationType(consumerType), resolvedAckMode.name()));
    }


    if (isXa(connection) && sourceConfiguration.getTransactionalAction() == ALWAYS_BEGIN) {
      messageConsumerDelegate =
          new JmsXaPollingMessageConsumerDelegate(connection, jmsSupport, destination,
                                                  consumerType, config, selector, sessionManager, connectionProvider, scheduler,
                                                  inboundContentType, inboundEncoding, sourceCallback, exceptionResolver);


    } else {
      messageConsumerDelegate =
          new JmsMessageListenerDelegate(new JmsMessageListenerFactory(resolvedAckMode, inboundEncoding, inboundContentType,
                                                                       config, sessionManager, jmsSupport,
                                                                       sourceCallback, connectionProvider, exceptionResolver),
                                         connection, jmsSupport, consumerType, destination, config, resolvedAckMode, selector,
                                         lockFactory, resourceCleaner, scheduler, jmsMaxIdleConnectionTimeout);
    }

    messageConsumerDelegate.createConsumers(numberOfConsumers);
  }

  private boolean isXa(JmsConnection connection) {
    return connection instanceof XaJmsTransactionalConnection;
  }

  private void validateTransactionType(JmsTransactionalConnection connection) throws ConnectionException {
    switch (sourceConfiguration.getTransactionalAction()) {
      case ALWAYS_BEGIN: {
        switch (sourceConfiguration.getTransactionType()) {
          case XA: {
            if (!(connection instanceof XATransactionalConnection)) {
              throw new ConnectionException(format("Invalid configuration, The message listener on the flow '%s' has " +
                  "been configured to work with XA Transactions, but the given connection from the config '%s' doesn't support it.\n"
                  +
                  "This can be fixed doing one of the following:\n" +
                  " - To work with Local transactions, select the 'LOCAL' Transaction Type on the Advanced Source Configuration \n"
                  +
                  " - To work with XA Transactions, enable XA in the connection configuration",
                                                   sourceConfiguration.getFlowName(),
                                                   sourceConfiguration.getConfigName()));
            }
            return;
          }
          case LOCAL: {
            if ((connection instanceof XATransactionalConnection)) {
              throw new ConnectionException(format("Invalid configuration: The message listener on the flow '%s' has " +
                  "been configured to work with Local Transactions, but the given connection from the config '%s' requires XA Transactions. \n"
                  +
                  "This can be fixed doing one of the following:\n" +
                  " - To work with XA Transactions, select the 'XA' Transaction Type on the Advanced Source Configuration\n" +
                  " - To work with Local transactions, disable XA in the connection configuration",
                                                   sourceConfiguration.getFlowName(), sourceConfiguration.getConfigName()));
            }
          }
        }
        break;
      }
      case NONE: {
        if (connection instanceof XATransactionalConnection) {
          LOGGER.info("A XA Connection is being used in a non transactional context, this could led to unexpected behaviour");
        }
      }
    }
  }

  public void onStop() {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug(format("Stopping JMS Listener on destination [%s:%s]", getDestinationType(consumerType), destination));
    }

    try {
      if (messageConsumerDelegate != null) {
        messageConsumerDelegate.stop();
        messageConsumerDelegate = null;
      }

      if (connection != null && !(connection instanceof XaJmsTransactionalConnection)) {
        resourceCleaner.releaseConnection(connection.get());
        connectionProvider.disconnect(connection);
      }

    } finally {
      if (scheduler != null) {
        scheduler.stop();
      }
    }
  }

  public void disableConsumers() {
    if (messageConsumerDelegate != null) {
      messageConsumerDelegate.disableConsumers();
    }
  }

  public void restart() throws MuleException {
    onStop();
    onStart(sourceCallback);
  }

  public void onSuccess(JmsResponseMessageBuilder messageBuilder,
                        CorrelationInfo correlationInfo,
                        SourceCallbackContext callbackContext) {
    if (messageConsumerDelegate != null) {
      messageConsumerDelegate.onSuccess(callbackContext);
    }
    callbackContext.<Destination>getVariable(REPLY_TO_DESTINATION_VAR)
        .ifPresent(replyTo -> doReply(messageBuilder, callbackContext, replyTo,
                                      correlationInfo, new RequestReplyContext(messageBuilder.getRequestReplyPattern(),
                                                                               callbackContext
                                                                                   .<String>getVariable(JmsListener.MESSAGE_ID_VAR)
                                                                                   .orElse(null),
                                                                               callbackContext
                                                                                   .<String>getVariable(JmsListener.CORRELATION_ID_VAR)
                                                                                   .orElse(null))));

  }

  public void onError(Error error, SourceCallbackContext callbackContext) {
    if (messageConsumerDelegate != null) {
      messageConsumerDelegate.onError(callbackContext, error);
    } else {
      LOGGER.debug("A error occurred after the Source being stopped. {}", error);
    }
  }

  private void doReply(JmsResponseMessageBuilder messageBuilder,
                       SourceCallbackContext callbackContext, Destination replyTo,
                       CorrelationInfo correlationInfo, RequestReplyContext requestReplyPattern) {
    final boolean replyToTopic = replyDestinationIsTopic(replyTo);
    String destinationName;
    try {
      destinationName = replyToTopic ? ((Topic) replyTo).getTopicName() : ((Queue) replyTo).getQueueName();
    } catch (JMSException e) {
      LOGGER.error(format("An error occurred during reply. Failed to obtain the destination name: %s", e.getMessage()));
      notifyIfConnectionProblem(callbackContext, e, exceptionResolver);
      return;
    }

    JmsMessageProducer producer = null;
    JmsSession session = null;
    try {
      session = getSession(connection, replyToTopic);

      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug(format("Begin reply to destination [%s] of type [%s]", destinationName, replyToTopic ? TOPIC : QUEUE));
      }

      Message message = messageBuilder.build(connection.getJmsSupport(), messageBuilder.getSendCorrelationId(), correlationInfo,
                                             session.get(), config);

      applyRequestResponsePattern(messageBuilder, requestReplyPattern, message);

      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Message built, sending message to " + destinationName);
      }

      producer = connection.createProducer(session, replyTo, replyToTopic);
      producer.publish(message, messageBuilder);
    } catch (Exception e) {
      LOGGER.error(format("An error occurred during reply to destination [%s] of type [%s]: %s",
                          destinationName, replyToTopic ? TOPIC : QUEUE, e.getMessage()),
                   e);
      rollbackIfInTransaction(callbackContext);
      notifyIfConnectionProblem(callbackContext, e, exceptionResolver);
    } finally {
      releaseResources(session, isPartOfCurrentTx(session, connection, sessionManager), producer);
    }
  }

  private void rollbackIfInTransaction(SourceCallbackContext callbackContext) {
    if (callbackContext.getTransactionHandle().isTransacted()) {
      try {
        callbackContext.getTransactionHandle().rollback();
      } catch (TransactionException e) {
        LOGGER.error("Unable to rollback transaction after the error occurred during reply.", e);
      }
    }
  }

  private void applyRequestResponsePattern(JmsResponseMessageBuilder messageBuilder, RequestReplyContext requestReplyContext,
                                           Message message)
      throws JMSException {
    if (messageBuilder.getCorrelationId() == null) {
      switch (requestReplyContext.getPattern()) {
        case CORRELATION_ID:
          message.setJMSCorrelationID(requestReplyContext.getCorrelationId());
          break;
        case MESSAGE_ID:
          message.setJMSCorrelationID(requestReplyContext.getMessageId());
          break;
        default:
          break;
      }
    }
  }

  private JmsSession getSession(JmsTransactionalConnection connection, boolean isTopic) throws JMSException {
    java.util.Optional<JmsSession> transactedSession = sessionManager.getTransactedSession(connection);
    if (transactedSession.isPresent()) {
      return transactedSession.get();
    } else {
      return connection.createSession(AUTO, isTopic);
    }
  }

  private boolean replyDestinationIsTopic(Destination destination) {
    // TODO: MULE-11156 - take into account the special logic in 3.x for handling Weblogic 8.x and 9.x
    // see 'org.mule.transport.jms.weblogic.WeblogicJmsTopicResolver#topic'

    if (destination instanceof Topic && destination instanceof Queue
        && jmsSupport instanceof Jms102bSupport) {
      LOGGER.error(StringMessageUtils.getBoilerPlate(
                                                     "Destination implements both Queue and Topic "
                                                         + "while complying with JMS 1.0.2b specification. "
                                                         + "Please report your application server or JMS vendor name and version "
                                                         + "to http://www.mulesoft.org/jira"));
    }

    return destination instanceof Topic;
  }


  private void validateNumberOfConsumers(int numberOfConsumers) {
    if (numberOfConsumers < 1) {
      throw new IllegalArgumentException("Invalid number of consumers: [" + numberOfConsumers
          + "]. The number should be 1 or greater.");
    }

    if (numberOfConsumers > 1 && consumerType.topic()) {
      TopicConsumer topicConsumer = (TopicConsumer) consumerType;

      if (!isCapableOfMultiConsumersOnTopic(topicConsumer)) {
        throw new IllegalArgumentException("Destination [" + destination + "] is a topic, but [" + numberOfConsumers
            + "] receivers have been requested. This is only possible for 'shared' topic consumers, otherwise use 1.");
      }
    }
  }

  private boolean isCapableOfMultiConsumersOnTopic(TopicConsumer topicConsumer) {
    return jmsSupport.getSpecification().equals(JMS_2_0) && topicConsumer.isShared();
  }

  private class RequestReplyContext {

    RequestReplyPattern pattern;
    String messageId;
    String correlationId;

    public RequestReplyContext(RequestReplyPattern pattern, String messageId, String correlationId) {
      this.pattern = pattern;
      this.messageId = messageId;
      this.correlationId = correlationId;
    }

    public RequestReplyPattern getPattern() {
      return pattern;
    }

    public String getMessageId() {
      return messageId;
    }

    public String getCorrelationId() {
      return correlationId;
    }

    @Override
    public String toString() {
      return "RequestReplyContext{" +
          "pattern=" + pattern +
          ", messageId='" + messageId + '\'' +
          ", correlationId='" + correlationId + '\'' +
          '}';
    }
  }
}
