/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.jms.commons.internal.operation.publishconsume;

import static org.mule.jms.commons.internal.common.JmsCommons.closeQuietly;

import org.mule.jms.commons.internal.connection.session.JmsSession;
import org.mule.jms.commons.internal.publish.JmsMessageProducer;
import org.mule.runtime.api.scheduler.Scheduler;

import javax.jms.CompletionListener;
import javax.jms.Message;

/**
 * {@link CompletionListener} implementation that closes the given resources once the message has been received.
 *
 * @since 1.0.0
 */
public class ResourceCloserCompletionListener implements CompletionListener {

  private CompletionListener delegate;
  private JmsSession producerSession;
  private AutoCloseable producer;
  private final Scheduler scheduler;


  public ResourceCloserCompletionListener(CompletionListener delegate,
                                          JmsSession jmsSession,
                                          JmsMessageProducer producer,
                                          Scheduler scheduler) {
    this.delegate = delegate;
    this.producerSession = jmsSession;
    this.producer = producer;
    this.scheduler = scheduler;
  }

  @Override
  public void onCompletion(Message message) {
    scheduler.submit(this.closeResources());
    delegate.onCompletion(message);
  }

  @Override
  public void onException(Message message, Exception exception) {
    scheduler.submit(this.closeResources());
    delegate.onException(message, exception);
  }

  private Runnable closeResources() {
    return () -> {
      closeQuietly(producer);
      closeQuietly(producerSession);
    };
  }
}
