/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.connectors.restconnect.commons.api.connection.oauth;

import static org.mule.connectors.restconnect.commons.api.error.RestConnectError.UNAUTHORIZED;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.http.api.HttpHeaders.Names.AUTHORIZATION;

import org.mule.connectors.restconnect.commons.api.operation.HttpResponseAttributes;
import org.mule.connectors.restconnect.commons.api.connection.DefaultRestConnection;
import org.mule.connectors.restconnect.commons.api.error.RestConnectError;
import org.mule.connectors.restconnect.commons.internal.util.RestRequestBuilder;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.extension.api.connectivity.oauth.AccessTokenExpiredException;
import org.mule.runtime.extension.api.connectivity.oauth.OAuthState;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.auth.HttpAuthentication;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import java.io.InputStream;
import java.util.concurrent.CompletableFuture;

/**
 * Specialization of {@link DefaultRestConnection} for resources protected with OAuth, regardless of
 * the grant type.
 * <p>
 * The main features of this connection are that:
 *
 * <ul>
 * <li>Each instance can only serve one resource owner id. New instances need to be created for each</li>
 * <li>When the service providers responds with a 401 status code, the response {@link CompletableFuture} will be
 * completed with an {@link AccessTokenExpiredException} so that the SDK triggers the refresh token workflow.</li>
 * </ul>
 *
 * @since 1.0
 */
public class OAuthRestConnection extends DefaultRestConnection {

  private final OAuthState oauthState;
  private final String resourceOwnerId;

  /**
   * Creates a new instance
   *
   * @param baseUri            the service base uri
   * @param configName         the name of the config that owns this connection
   * @param httpClient         the client to use
   * @param authentication     the authentication mechanism to use, or {@code null}
   * @param defaultQueryParams query params to be automatically added to all requests done through this connection
   * @param defaultHeaders     headers to be automatically added to all requests done through this connection
   * @param oauthState         the {@link OAuthState} for the current connection
   * @param resourceOwnerId    the id of the resource owner for this connection
   */
  public OAuthRestConnection(String baseUri,
                             String configName,
                             HttpClient httpClient,
                             HttpAuthentication authentication,
                             MultiMap<String, String> defaultQueryParams,
                             MultiMap<String, String> defaultHeaders,
                             OAuthState oauthState,
                             String resourceOwnerId) {
    super(baseUri, configName, httpClient, authentication, defaultQueryParams, defaultHeaders);

    checkArgument(resourceOwnerId != null, "resourceOwnerId cannot be null");
    this.resourceOwnerId = resourceOwnerId;

    checkArgument(oauthState != null, "oauthState cannot be null");
    this.oauthState = oauthState;
  }

  @Override
  protected HttpRequest buildRequest(RestRequestBuilder requestBuilder) {
    requestBuilder.addHeader(AUTHORIZATION, "Bearer " + oauthState.getAccessToken());
    return super.buildRequest(requestBuilder);
  }

  @Override
  protected void handleResponseError(HttpResponse response,
                                     MediaType defaultResponseMediaType,
                                     CompletableFuture<Result<InputStream, HttpResponseAttributes>> future,
                                     StreamingHelper streamingHelper,
                                     RestConnectError error) {
    if (error == UNAUTHORIZED) {
      future.completeExceptionally(new AccessTokenExpiredException(resourceOwnerId));
    } else {
      super.handleResponseError(response, defaultResponseMediaType, future, streamingHelper, error);
    }
  }
}
