/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.connectors.restconnect.commons.api.connection.validation;

import static java.lang.String.format;
import static java.nio.charset.Charset.defaultCharset;
import static org.mule.connectors.restconnect.commons.internal.RestConnectConstants.ATTRIBUTES_VAR;
import static org.mule.connectors.restconnect.commons.internal.RestConnectConstants.PAYLOAD_VAR;
import static org.mule.connectors.restconnect.commons.internal.util.ConnectionValidationUtils.connectionExceptionResult;
import static org.mule.connectors.restconnect.commons.internal.util.RestConnectUtils.consumeStringAndClose;
import static org.mule.connectors.restconnect.commons.internal.util.RestConnectUtils.isBlank;
import static org.mule.connectors.restconnect.commons.internal.util.RestConnectUtils.isNotBlank;
import static org.mule.runtime.api.connection.ConnectionValidationResult.success;
import static org.mule.runtime.api.metadata.DataType.BOOLEAN;
import static org.mule.runtime.api.metadata.DataType.STRING;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JSON;
import static org.mule.runtime.api.metadata.MediaType.parse;
import static org.mule.runtime.api.metadata.TypedValue.of;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import org.mule.connectors.restconnect.commons.api.operation.HttpResponseAttributes;
import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.el.ValidationResult;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.core.api.expression.ExpressionRuntimeException;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import java.nio.charset.Charset;
import java.util.List;
import java.util.Set;
import java.util.StringJoiner;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Evaluates a HTTP response and determines if a connection is valid of not.
 *
 * @since 1.0
 */
public final class ConnectionValidator {

  private static final Logger LOGGER = LoggerFactory.getLogger(ConnectionValidator.class);

  /**
   * Validates a HTTPResponse against the provided connection validation settings.
   * @param response  The server's HTTP response that needs to be validated.
   * @param settings  The validation settings for the provided response.
   * @return  A {@link ConnectionValidationResult} indicating if the connection is valid or not and why.
   */
  public static ConnectionValidationResult validateConnectionResponse(HttpResponse response,
                                                                      ConnectionValidationSettings settings) {
    return validateConnectionResponse(response,
                                      settings.getValidStatusCodes(),
                                      settings.getExpressionLanguage(),
                                      settings.getTestConnectionValidations(),
                                      settings.getResponseMediaType());
  }

  /**
   * Validates the server response and build a validation result indicating if that evaluation was successful or not.
   * @param response                the test connection request's response.
   * @param validStatusCodes        a set containing all the response tatus codes that will be considered as valid.
   * @param expressionLanguage      an expression language that will be used to execute the provided expression.
   * @param testConnectionValidations   the expressions that will be evaluated to check if the server response is valid.
   * @param responseMediaType       if provided the media type provided by the server will be ignored and this will be used instead.
   *                                This parameter should be sent if it is known that the server does not return a content-type header.
   * @return a {@link ConnectionValidationResult} that indicates if this connection is valid or not.
   */
  private static ConnectionValidationResult validateConnectionResponse(HttpResponse response,
                                                                       Set<Integer> validStatusCodes,
                                                                       ExpressionLanguage expressionLanguage,
                                                                       List<TestConnectionValidation> testConnectionValidations,
                                                                       MediaType responseMediaType) {

    if (validStatusCodes.stream().noneMatch(x -> x.equals(response.getStatusCode()))) {
      final StringJoiner statusCodeJoiner = new StringJoiner(", ");
      validStatusCodes.forEach(x -> statusCodeJoiner.add(x.toString()));

      return connectionExceptionResult(format("Server responded with status code %s. Expected status code were: [%s]",
                                              response.getStatusCode(),
                                              statusCodeJoiner.toString()));
    }

    BindingContext context = buildValidationResponseContext(response, responseMediaType);

    for (TestConnectionValidation expression : testConnectionValidations) {
      ConnectionValidationResult validationResult = validateExpression(context, expressionLanguage, expression);
      if (!validationResult.isValid()) {
        return validationResult;
      }
    }

    return success();
  }

  /**
   * Evaluates a {@link TestConnectionValidation} and returns a validation result indicating if it was successful or not.
   * If the validation expression provides an error template expression it will be used to generate the error message.
   * @param bindingContext        a binding context containing the corresponding server response.
   * @param expressionLanguage    an expression language that will be used to execute the provided expression.
   * @param testConnectionValidation  the expression that will be evaluated.
   * @return a {@link ConnectionValidationResult} that indicates if this validation result is successful or not.
   */
  private static ConnectionValidationResult validateExpression(BindingContext bindingContext,
                                                               ExpressionLanguage expressionLanguage,
                                                               TestConnectionValidation testConnectionValidation) {

    String validationExp = testConnectionValidation.getValidationExpression();

    if (isNotBlank(validationExp)) {
      ValidationResult validationResult = expressionLanguage.validate(validationExp);
      if (!validationResult.isSuccess()) {
        return connectionExceptionResult(format("Validation expression is not valid. %s", validationExp));
      }

      try {
        TypedValue<?> expressionResult = expressionLanguage.evaluate(validationExp, BOOLEAN, bindingContext);

        if (!expressionResult.getValue().equals(true)) {
          String errorMessage = tryGetErrorMessage(bindingContext, expressionLanguage, testConnectionValidation);

          return isBlank(errorMessage)
              ? connectionExceptionResult(format("Expression evaluation did not return true. Expression: %s", validationExp))
              : connectionExceptionResult(errorMessage);
        }
      } catch (ExpressionRuntimeException e) {
        return connectionExceptionResult(format("Runtime error evaluating expression: %s", validationExp), e);
      }
    }

    return success();
  }

  /**
   * Tries to build an error message using the provided error template expression.
   * @param bindingContext        a binding context containing the corresponding server response.
   * @param expressionLanguage    an expression language that will be used to execute the provided expression.
   * @param testConnectionValidation  the expression that will be used to build the error message.
   *                              This expression should return an string.
   * @return An string containing the generated error message. Null if the error message can not be built.
   */
  private static String tryGetErrorMessage(BindingContext bindingContext,
                                           ExpressionLanguage expressionLanguage,
                                           TestConnectionValidation testConnectionValidation) {
    String errorExp = testConnectionValidation.getErrorTemplateExpression();

    if (isNotBlank(errorExp)) {
      try {
        TypedValue<?> expressionResult = expressionLanguage.evaluate(errorExp, STRING, bindingContext);
        return expressionResult.getValue().toString();
      } catch (Throwable t) {
        LOGGER.error("Error evaluating error template expression", t);
      }
    }
    return null;
  }

  /**
   * Builds a binding context containing the response body in the payload and the response attributes in the attributes variable.
   * @param response          the test connection response from the server.
   * @param overrideMediaType if provided the media type provided by the server will be ignored and this will be used instead.
   *                          This parameter should be sent if it is known that the server does not return a content-type header.
   * @return A binding context
   */
  private static BindingContext buildValidationResponseContext(HttpResponse response, MediaType overrideMediaType) {
    MediaType defaultMediaType =
        response.containsHeader(CONTENT_TYPE) ? parse(response.getHeaderValue(CONTENT_TYPE)) : APPLICATION_JSON;

    MediaType mediaType = overrideMediaType != null ? overrideMediaType : defaultMediaType;
    Charset charset = mediaType.getCharset().orElse(defaultCharset());

    return BindingContext.builder()
        .addBinding(PAYLOAD_VAR, consumeStringAndClose(response.getEntity().getContent(), mediaType, charset))
        .addBinding(ATTRIBUTES_VAR, of(toAttributes(response)))
        .build();
  }

  private static HttpResponseAttributes toAttributes(HttpResponse response) {
    return new HttpResponseAttributes(response.getStatusCode(), response.getReasonPhrase(), response.getHeaders());
  }
}
