/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.connectors.restconnect.commons.api.operation.paging;

import org.mule.connectors.restconnect.commons.api.connection.RestConnection;
import org.mule.connectors.restconnect.commons.api.operation.HttpResponseAttributes;
import org.mule.connectors.restconnect.commons.internal.util.RestRequestBuilder;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;

import java.util.List;
import java.util.function.Function;

/**
 * Implementation of {@link RestPagingProvider} for APIs which do paging based on requesting individual pages by number
 *
 * @since 1.0
 */
public class PageNumberPagingProvider extends RestPagingProvider {

  private final String pageNumberParamName;
  private int pageNumber;

  /**
   * Creates a new instance
   *
   * @param pageNumberParamName the name of the query param that carries the page number value
   * @param initialPageNumber   the initial page to get in the first request
   * @param requestFactory      a {@link Function} to generate the request to be used on each page request. Each invocation should yield a different instance
   * @param expressionLanguage  the app's {@link ExpressionLanguage}
   * @param streamingHelper     the {@link StreamingHelper} associated to the executing operation
   * @param payloadExpression   a DW expression to extract the data from the response
   * @param defaultMediaType    the {@link MediaType} for the page items if the server doesn't specify one
   * @param responseTimeout     the timeout for each request
   */
  public PageNumberPagingProvider(String pageNumberParamName,
                                  int initialPageNumber,
                                  Function<RestConnection, RestRequestBuilder> requestFactory,
                                  ExpressionLanguage expressionLanguage,
                                  StreamingHelper streamingHelper,
                                  String payloadExpression,
                                  MediaType defaultMediaType,
                                  int responseTimeout) {
    super(requestFactory, expressionLanguage, streamingHelper, payloadExpression, defaultMediaType, responseTimeout);
    this.pageNumberParamName = pageNumberParamName;
    pageNumber = initialPageNumber;
  }

  @Override
  protected void configureRequest(RestRequestBuilder requestBuilder) {
    requestBuilder.addQueryParam(pageNumberParamName, String.valueOf(pageNumber));
  }

  @Override
  protected void onPage(List<TypedValue<String>> page, TypedValue<String> rawPage,
                        HttpResponseAttributes httpResponseAttributes) {
    pageNumber++;
  }
}
