/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.connectors.restconnect.commons.api.datasense.resolver;

import static org.mule.metadata.api.builder.BaseTypeBuilder.create;

import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.metadata.resolving.OutputStaticTypeResolver;
import org.mule.runtime.core.api.util.IOUtils;

import java.io.InputStream;

/**
 * An {@link OutputStaticTypeResolver} to use in paging operations. This resolver takes the path to a schema
 * shipped with the connector and generates an {@link ArrayType} which target type is generated by parsing that schema.
 *
 * @since 1.0
 */
public abstract class PagingTypeResolver extends OutputStaticTypeResolver {

  /**
   * Absolute path to the schema between the context of the owning connector. This path will be read
   * using the {@link Class#getResourceAsStream(String)} method
   *
   * @return the schema path inside the connector
   */
  protected abstract String getSchemaPath();

  /**
   * Parses the given {@code schema} and transforms it to a {@link MetadataType}
   * @param schema the schema to parse
   * @return a {@link MetadataType}
   */
  protected abstract MetadataType loadSchema(String schema);

  @Override
  public MetadataType getStaticMetadata() {
    return create(MetadataFormat.JAVA).arrayType().of(loadSchema(readSchema())).build();
  }

  private String readSchema() {
    String schema;
    String schemaPath = getSchemaPath();
    try (InputStream in = getClass().getResourceAsStream(getSchemaPath())) {
      if (in == null) {
        throw new IllegalArgumentException("Could not find schema at " + schemaPath);
      }

      schema = IOUtils.toString(in);
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
    return schema;
  }
}
