/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.connectors.restconnect.commons.api.operation.paging;

import static java.lang.Thread.currentThread;
import static java.util.Collections.emptyList;
import static java.util.Optional.empty;
import static java.util.OptionalLong.of;
import static org.mule.connectors.restconnect.commons.internal.util.RestConnectUtils.consumeStringAndClose;
import static org.mule.connectors.restconnect.commons.internal.util.RestConnectUtils.isBlank;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.metadata.DataType.fromType;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.connectors.restconnect.commons.api.connection.RestConnection;
import org.mule.connectors.restconnect.commons.api.operation.HttpResponseAttributes;
import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.streaming.PagingProvider;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;

import java.io.InputStream;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.function.Function;

import org.slf4j.Logger;

/**
 * Provides the base functionality for paginating the results of a REST endpoint.
 * <p>
 * Different APIs use different paging strategies which are tackled in implementations of this class. This class provides
 * the basics as:
 *
 * <ul>
 * <li>Executing one request per page to be obtained</li>
 * <li>Ability to parse the response to extract the data and useful metadata as the total results count (if provided)</li>
 * <li>Handling streaming</li>
 * <li>Handling the nuances of the {@link PagingProvider} contract</li>
 * <li>Handling encodings and media types</li>
 * </ul>
 * <p>
 * Each item in the obtained page will be returned as an individual {@link TypedValue}, with the proper mime type and encoding
 * set.
 *
 * @since 1.0
 */
public abstract class RestPagingProvider implements PagingProvider<RestConnection, TypedValue<String>> {

  private static final DataType INTEGER_DATA_TYPE = fromType(Integer.class);
  private static final Logger LOGGER = getLogger(RestPagingProvider.class);

  protected final ExpressionLanguage expressionLanguage;
  protected final int responseTimeout;
  protected final MediaType mediaType;
  protected final String encoding;

  private final Function<RestConnection, HttpRequestBuilder> requestFactory;
  private final StreamingHelper streamingHelper;
  private final String payloadExpression;
  private final DataType dataType;

  private int pageIndex = -1;
  private boolean stopPaging = false;

  /**
   * Creates a new instance
   *
   * @param requestFactory         a {@link Function} to generate the request to be used on each page request. Each invocation should yield a different instance
   * @param expressionLanguage     the app's {@link ExpressionLanguage}
   * @param streamingHelper        the {@link StreamingHelper} associated to the executing operation
   * @param payloadExpression      a DW expression to extract the data from the response
   * @param mediaType              the {@link MediaType} for the page items
   * @param encoding               the encoding for the page items
   * @param responseTimeout        the timeout for each request
   */
  public RestPagingProvider(Function<RestConnection, HttpRequestBuilder> requestFactory,
                            ExpressionLanguage expressionLanguage,
                            StreamingHelper streamingHelper,
                            String payloadExpression,
                            MediaType mediaType, String encoding,
                            int responseTimeout) {

    this.requestFactory = requestFactory;
    this.streamingHelper = streamingHelper;
    this.expressionLanguage = expressionLanguage;
    this.responseTimeout = responseTimeout;
    this.payloadExpression = payloadExpression;
    this.mediaType = mediaType;
    this.encoding = encoding;
    dataType = DataType.builder().type(String.class).mediaType(mediaType).charset(encoding).build();
  }

  /**
   * Depending on the paging strategy, different parameters or headers may need to be passed. This method
   * receives the raw output of the {@link #requestFactory} in order to configure the custom parameters that apply/
   *
   * @param requestBuilder a {@link HttpRequestBuilder}
   */
  protected abstract void configureRequest(HttpRequestBuilder requestBuilder);

  @Override
  public final List<TypedValue<String>> getPage(RestConnection connection) {
    if (stopPaging) {
      return emptyList();
    }

    final HttpRequestBuilder requestBuilder = requestFactory.apply(connection);
    configureRequest(requestBuilder);

    Result<InputStream, HttpResponseAttributes> result;
    try {
      result = connection.request(requestBuilder, responseTimeout, streamingHelper).get();
    } catch (ExecutionException e) {
      Throwable cause = e.getCause();
      if (cause instanceof RuntimeException) {
        throw (RuntimeException) cause;
      }

      throw new MuleRuntimeException(createStaticMessage(cause.getMessage()), cause);
    } catch (InterruptedException e) {
      currentThread().interrupt();
      throw new MuleRuntimeException(e);
    }

    final String rawPage = consumeStringAndClose(result.getOutput(), encoding);

    if (stopPaging) {
      return emptyList();
    }

    List<TypedValue<String>> page;

    if (isBlank(rawPage)) {
      page = emptyList();
    } else {
      page = extractPayload(rawPage);
      pageIndex++;
    }

    onPage(page, rawPage);

    return page;
  }

  @Override
  public Optional<Integer> getTotalResults(RestConnection connection) {
    return empty();
  }

  @Override
  public final void close(RestConnection connection) throws MuleException {
    doClose(connection);
  }

  /**
   * Invoked each time a page is obtained.
   *
   * @param page the obtained page
   */
  protected void onPage(List<TypedValue<String>> page, String rawPage) {
    // empty default implementation
  }

  protected void stopPaging() {
    stopPaging = true;
  }

  /**
   * Invoked when {@link #close(RestConnection)} is invoked
   *
   * @param connection the current connection
   * @throws MuleException
   */
  protected void doClose(RestConnection connection) throws MuleException {
    // empty default implementation
  }

  protected <T> TypedValue<T> evaluate(String content, String expression, DataType expectedOutputType) {
    return (TypedValue<T>) expressionLanguage.evaluate(expression, expectedOutputType, toBindingContext(content));
  }

  private List<TypedValue<String>> extractPayload(String content) {
    TypedValue<?> payload = expressionLanguage.evaluate(payloadExpression, toBindingContext(content));
    Iterator<TypedValue<?>> it = expressionLanguage.split("#[payload default []]", toBindingContext(payload));
    return toList(it);
  }

  private BindingContext toBindingContext(String content) {
    return toBindingContext(new TypedValue<>(content, dataType, of(content.length())));
  }

  private List<TypedValue<String>> toList(Iterator<TypedValue<?>> iterator) {
    List<TypedValue<String>> list = new LinkedList<>();

    while (iterator.hasNext()) {
      String value = consumeStringAndClose(iterator.next().getValue(), encoding);
      list.add(new TypedValue<>(value, dataType, of(value.length())));
    }

    return list;
  }

  private BindingContext toBindingContext(TypedValue<?> content) {
    return BindingContext.builder()
        .addBinding("payload", content)
        .build();
  }
}
