/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.connectors.restconnect.commons.api.error;

import static java.util.Optional.ofNullable;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.getStatusByCode;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import org.mule.runtime.extension.api.error.ErrorTypeDefinition;
import org.mule.runtime.extension.api.error.MuleErrors;
import org.mule.runtime.http.api.HttpConstants.HttpStatus;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;

import java.util.Optional;
import java.util.function.Function;

/**
 * Represents an error that can happen in an HTTP operation.
 *
 * @since 1.0
 */
public enum RestConnectError implements ErrorTypeDefinition<RestConnectError> {

  CONNECTIVITY(MuleErrors.CONNECTIVITY), CLIENT_ERROR, TIMEOUT(CLIENT_ERROR), UNAUTHORIZED(CLIENT_ERROR), NOT_FOUND(
      CLIENT_ERROR), TOO_MANY_REQUESTS(CLIENT_ERROR), BAD_REQUEST(CLIENT_ERROR), UNSUPPORTED_MEDIA_TYPE(CLIENT_ERROR,
          request -> "media type " +
              request.getHeaderValue(CONTENT_TYPE) + " not supported"), NOT_ACCEPTABLE,


  SERVER_ERROR, INTERNAL_SERVER_ERROR(SERVER_ERROR), SERVICE_UNAVAILABLE(SERVER_ERROR);

  private ErrorTypeDefinition<?> parentErrorType;
  private Function<HttpRequest, String> errorMessageFunction;

  RestConnectError() {
    String message = this.name().replace("_", " ").toLowerCase();
    errorMessageFunction = httpRequest -> message;
  }

  RestConnectError(ErrorTypeDefinition<?> parentErrorType) {
    this();
    this.parentErrorType = parentErrorType;
  }

  RestConnectError(ErrorTypeDefinition<?> parentErrorType, Function<HttpRequest, String> errorMessageFunction) {
    this.parentErrorType = parentErrorType;
    this.errorMessageFunction = errorMessageFunction;
  }

  @Override
  public Optional<ErrorTypeDefinition<? extends Enum<?>>> getParent() {
    return ofNullable(parentErrorType);
  }

  /**
   * Returns the {@link RestConnectError} corresponding to a given status code. A match is found if there's an
   * {@link RestConnectError} with the same name as the status code's corresponding {@link HttpStatus}.
   *
   * @param statusCode the HTTP status code to search for
   * @return an {@link Optional} with the error that matches the {@code statusCode}
   */
  public static Optional<RestConnectError> getErrorByCode(int statusCode) {
    RestConnectError error = null;
    HttpStatus status = getStatusByCode(statusCode);
    if (status != null) {
      try {
        error = RestConnectError.valueOf(status.name());
      } catch (IllegalArgumentException e) {
        // do nothing. Will handle this in the next if block
      }
    }

    if (error == null) {
      if (statusCode >= 400 && statusCode < 500) {
        error = CLIENT_ERROR;
      } else if (statusCode >= 500) {
        error = SERVER_ERROR;
      }
    }

    return ofNullable(error);
  }

  /**
   * Returns an {@link HttpStatus} corresponding to a given {@link RestConnectError}. A match is found if there's an
   * {@link HttpStatus} with the same name as the {@link RestConnectError}.
   *
   * @param error the {@link RestConnectError} to match
   * @return an {@link Optional} with the status that matches the {@code error}
   */
  public static Optional<HttpStatus> getHttpStatus(RestConnectError error) {
    HttpStatus result = null;
    for (HttpStatus status : HttpStatus.values()) {
      if (error.name().equals(status.name())) {
        result = status;
      }
    }
    return ofNullable(result);
  }

  /**
   * Returns the custom error message for this {@link RestConnectError} based on the {@link HttpRequest} that triggered it.
   *
   * @param request the {@link HttpRequest} that caused the error
   * @return the custom error message associated to the error
   */
  public String getErrorMessage(HttpRequest request) {
    return errorMessageFunction.apply(request);
  }

}
