/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.connectors.restconnect.commons.api.operation;

import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.runtime.http.api.HttpHeaders.Names.TRANSFER_ENCODING;
import static org.mule.runtime.http.api.HttpHeaders.Values.CHUNKED;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.connectors.restconnect.commons.api.connection.RestConnection;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.mule.runtime.http.api.HttpConstants.Method;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;

import java.io.InputStream;
import java.util.Optional;
import java.util.function.BiConsumer;

import org.slf4j.Logger;

/**
 * Base class for operations that consume a remote REST endpoint
 *
 * @since 1.0
 */
public abstract class BaseRestConnectOperation {

  private static final Logger LOGGER = getLogger(BaseRestConnectOperation.class);

  /**
   * Performs the request using the given {@code builder} and {@code connection} and completes
   * the {@code callback} accordingly.
   *
   * @param connection            the connection to use
   * @param builder               the request builder
   * @param responseTimeoutMillis the request timeout in millis
   * @param streamingHelper       the {@link StreamingHelper}
   * @param callback              the operation's {@link CompletionCallback}
   */
  protected void doRequest(RestConnection connection,
                           HttpRequestBuilder builder,
                           int responseTimeoutMillis,
                           StreamingHelper streamingHelper,
                           CompletionCallback<InputStream, HttpResponseAttributes> callback) {

    try {
      connection.request(builder, responseTimeoutMillis, streamingHelper).whenComplete(handleResponse(callback));
    } catch (Throwable t) {
      callback.error(t);
    }
  }

  /**
   * Performs the request using the given {@code builder} and {@code connection} and completes
   * the {@code callback} accordingly, expecting the response to not contain a body.
   *
   * If successful, the callback will be completed with an empty String as payload.
   *
   * @param connection            the connection to use
   * @param builder               the request builder
   * @param responseTimeoutMillis the request timeout in millis
   * @param streamingHelper       the {@link StreamingHelper}
   * @param callback              the operation's {@link CompletionCallback}
   */
  protected void doVoidRequest(RestConnection connection,
                               HttpRequestBuilder builder,
                               int responseTimeoutMillis,
                               StreamingHelper streamingHelper,
                               CompletionCallback<String, HttpResponseAttributes> callback) {

    try {
      connection.bodylessRequest(builder, responseTimeoutMillis, streamingHelper).whenComplete(handleResponse(callback));
    } catch (Throwable t) {
      callback.error(t);
    }
  }

  private <T> BiConsumer<Result<T, HttpResponseAttributes>, Throwable> handleResponse(
                                                                                      CompletionCallback<T, HttpResponseAttributes> callback) {

    return (r, e) -> {
      if (e != null) {
        callback.error(e);
      } else {
        callback.success(r);
      }
    };
  }


  /**
   * Creates a preconfigured {@link HttpRequestBuilder} per the given parameters.
   * <p>
   * This method automatically merges and configures query params and headers per the config's default
   * and the operation's specific parameters. It also handles the {@code TRANSFER_ENCODING} and {@code CONTENT-LENGTH}
   * parameters based on the provided headers and {@code body}.
   * <p>
   * The returned builder can still to add/change additional information.
   *
   * @param baseUri    the base uri to be used
   * @param path       the path of the resource to request, relative to the base URI
   * @param method     the request method
   * @param body       a nullable {@link TypedValue} to be sent as the request entity
   * @param parameters the request parameters
   * @return a new {@link HttpRequestBuilder}
   */
  protected HttpRequestBuilder requestBuilder(String baseUri,
                                              String path,
                                              Method method,
                                              TypedValue<InputStream> body,
                                              RequestParameters parameters) {

    final String uri = buildRequestUri(baseUri, path);
    HttpRequestBuilder builder = HttpRequest.builder().method(method).uri(uri);

    builder.headers(parameters.getCustomHeaders());
    builder.queryParams(parameters.getCustomQueryParams());

    if (body != null) {

      setContentLengthHeader(body, uri, builder);
      setContentType(body, builder);

      if (body.getValue() != null) {
        builder.entity(new InputStreamHttpEntity(body.getValue()));
      }
    }

    return builder;
  }

  private void setContentType(TypedValue<InputStream> body, HttpRequestBuilder builder) {
    if (!builder.getHeaderValue(CONTENT_TYPE).isPresent()) {
      MediaType mediaType = body.getDataType().getMediaType();
      if (mediaType != null && !mediaType.getPrimaryType().equals("*")) {
        builder.addHeader(CONTENT_TYPE, mediaType.toRfcString());
      }
    }
  }

  private void setContentLengthHeader(TypedValue<InputStream> body, String uri, HttpRequestBuilder builder) {
    Optional<String> customLength = builder.getHeaderValue(CONTENT_LENGTH);
    boolean isChunked = builder.getHeaderValue(TRANSFER_ENCODING).map(CHUNKED::equals).orElse(false);

    if (body.getByteLength().isPresent()) {
      boolean addHeader = true;
      String length = String.valueOf(body.getByteLength().getAsLong());
      if (customLength.isPresent()) {
        LOGGER.warn("Invoking URI {} with body of known length {}. However, a {} header with value {} was manually specified. "
            + "Will proceed with the custom value.",
                    uri, length, CONTENT_LENGTH, customLength.get());

        addHeader = false;
      }

      if (isChunked) {
        LOGGER.debug("Invoking URI {} with a manually set {}: {} header, even though body is of known length {}. "
            + "Skipping automatic addition of {} header", uri, TRANSFER_ENCODING, CHUNKED, length, CONTENT_LENGTH);
        addHeader = false;
      }

      if (addHeader) {
        builder.addHeader(CONTENT_LENGTH, length);
      }

    } else if (!isChunked && !customLength.isPresent()) {
      builder.addHeader(TRANSFER_ENCODING, CHUNKED);
    }
  }

  private String buildRequestUri(String baseUri, String path) {
    boolean pathStartsWithSlash = path.startsWith("/");
    boolean baseEndsInSlash = baseUri.endsWith("/");

    if (pathStartsWithSlash && baseEndsInSlash) {
      path = path.substring(1);
    } else if (!pathStartsWithSlash && !baseEndsInSlash) {
      baseUri += '/';
    }

    return baseUri + path;
  }
}
