/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.connectors.restconnect.commons.api.operation.paging;

import org.mule.connectors.restconnect.commons.api.connection.RestConnection;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;

import java.util.List;
import java.util.function.Function;

/**
 * Implementation of {@link RestPagingProvider} for APIs which do paging based on an offset query param.
 *
 * @since 1.0
 */
public class OffsetPagingProvider extends RestPagingProvider {

  private final String offsetParamName;
  private int offset;

  /**
   * Creates a new instance
   *
   * @param offsetParamName        the name of the query param that carries the offset value
   * @param initialOffset          the initial offset to apply on the first request
   * @param requestFactory         a {@link Function} to generate the request to be used on each page request. Each invocation should yield a different instance
   * @param expressionLanguage     the app's {@link ExpressionLanguage}
   * @param streamingHelper        the {@link StreamingHelper} associated to the executing operation
   * @param payloadExpression      a DW expression to extract the data from the response
   * @param mediaType              the {@link MediaType} for the page items
   * @param encoding               the encoding for the page items
   * @param responseTimeout        the timeout for each request
   */
  public OffsetPagingProvider(String offsetParamName,
                              int initialOffset,
                              Function<RestConnection, HttpRequestBuilder> requestFactory,
                              ExpressionLanguage expressionLanguage,
                              StreamingHelper streamingHelper,
                              String payloadExpression,
                              MediaType mediaType,
                              String encoding,
                              int responseTimeout) {
    super(requestFactory, expressionLanguage, streamingHelper, payloadExpression, mediaType, encoding, responseTimeout);
    this.offsetParamName = offsetParamName;
    offset = initialOffset;
  }

  @Override
  protected void configureRequest(HttpRequestBuilder requestBuilder) {
    requestBuilder.addQueryParam(offsetParamName, String.valueOf(offset));
  }

  @Override
  protected void onPage(List<TypedValue<String>> page, String rawPage) {
    offset += page.size();
  }
}
