/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.connectors.restconnect.commons.api.connection;

import org.mule.connectors.restconnect.commons.api.operation.HttpResponseAttributes;
import org.mule.connectors.restconnect.commons.api.error.RequestException;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;

import java.io.InputStream;
import java.util.concurrent.CompletableFuture;

/**
 * Generic contract for connections that allow consuming a Rest API by the use of an underlying {@link HttpClient}
 *
 * @since 1.0
 */
public interface RestConnection {

  /**
   * Sends a request and returns a {@link CompletableFuture} with a {@link Result} which output is the response body in
   * the form of an {@link InputStream} and a {@link HttpResponseAttributes}.
   * <p>
   * In case of failure, the future will be exceptionally completed with a {@link RequestException}, which
   * {@link RequestException#getErrorMessage()} will have its payload made repeatable through the use of the
   * {@code streamingHelper}
   *
   * @param requestBuilder        the {@link HttpRequestBuilder} on which the request has been configured
   * @param responseTimeoutMillis the request timeout in milliseconds
   * @param streamingHelper       the {@link StreamingHelper} in case an error message needs to have its payload made repeatable
   * @return a {@link CompletableFuture}
   */
  CompletableFuture<Result<InputStream, HttpResponseAttributes>> request(HttpRequestBuilder requestBuilder,
                                                                         int responseTimeoutMillis,
                                                                         MediaType defaultResponseMediaType,
                                                                         StreamingHelper streamingHelper);


  /**
   * Sends a request for which a response entity is not expected. It returns a {@link CompletableFuture} with
   * a {@link Result} which holds an empty String as output and a {@link HttpResponseAttributes}.
   * <p>
   * In case of failure, the future will be exceptionally completed with a {@link RequestException}, which
   * {@link RequestException#getErrorMessage()} will have its payload made repeatable through the use of the
   * {@code streamingHelper}
   *
   * @param requestBuilder        the {@link HttpRequestBuilder} on which the request has been configured
   * @param responseTimeoutMillis the request timeout in milliseconds
   * @param streamingHelper       the {@link StreamingHelper} in case an error message needs to have its payload made repeatable
   * @return a {@link CompletableFuture}
   */
  CompletableFuture<Result<String, HttpResponseAttributes>> bodylessRequest(HttpRequestBuilder requestBuilder,
                                                                            int responseTimeoutMillis,
                                                                            MediaType defaultResponseMediaType,
                                                                            StreamingHelper streamingHelper);

  /**
   * @return the base uri of the REST API being consumed
   */
  String getBaseUri();

  /**
   * Stops this connection by freeing resources and stopping the underlying {@link HttpClient}
   */
  void stop();

}
