/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.connectors.restconnect.commons.api.operation;

import static java.lang.String.valueOf;
import static org.mule.connectors.restconnect.commons.api.configuration.StreamingType.ALWAYS;
import static org.mule.connectors.restconnect.commons.api.configuration.StreamingType.NEVER;
import static org.mule.runtime.api.metadata.MediaType.ANY;
import static org.mule.runtime.core.api.util.IOUtils.closeQuietly;
import static org.mule.runtime.core.api.util.IOUtils.toByteArray;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.runtime.http.api.HttpHeaders.Names.TRANSFER_ENCODING;
import static org.mule.runtime.http.api.HttpHeaders.Values.CHUNKED;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.connectors.restconnect.commons.api.configuration.RestConnectConfiguration;
import org.mule.connectors.restconnect.commons.api.configuration.StreamingType;
import org.mule.connectors.restconnect.commons.api.connection.RestConnection;
import org.mule.connectors.restconnect.commons.api.multipart.MultipartPayloadBuilder;
import org.mule.connectors.restconnect.commons.internal.multipart.DWMultipartPayloadBuilder;
import org.mule.connectors.restconnect.commons.internal.util.CloserCompletionCallbackDecorator;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.mule.runtime.http.api.HttpConstants.Method;
import org.mule.runtime.http.api.domain.entity.ByteArrayHttpEntity;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;

import java.io.InputStream;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

import javax.inject.Inject;

import org.slf4j.Logger;

/**
 * Base class for operations that consume a remote REST endpoint
 *
 * @since 1.0
 */
public abstract class BaseRestConnectOperation {

  private static final Logger LOGGER = getLogger(BaseRestConnectOperation.class);

  @Inject
  private ExpressionLanguage expressionLanguage;

  /**
   * Performs the request using the given {@code builder} and {@code connection} and completes
   * the {@code callback} accordingly.
   *
   * @param config                the operation's config
   * @param connection            the connection to use
   * @param builder               the request builder
   * @param responseTimeoutMillis the request timeout in millis
   * @param streamingHelper       the {@link StreamingHelper}
   * @param callback              the operation's {@link CompletionCallback}
   */
  protected void doRequest(RestConnectConfiguration config,
                           RestConnection connection,
                           HttpRequestBuilder builder,
                           int responseTimeoutMillis,
                           StreamingHelper streamingHelper,
                           CompletionCallback<InputStream, HttpResponseAttributes> callback) {

    try {
      connection.request(builder, responseTimeoutMillis, resolveDefaultResponseMediaType(config), streamingHelper)
          .whenComplete(handleResponse(callback));
    } catch (Throwable t) {
      callback.error(t);
    }
  }

  protected MediaType resolveDefaultResponseMediaType(RestConnectConfiguration config) {
    MediaType mediaType = getDefaultResponseMediaType();
    if (!mediaType.getCharset().isPresent()) {
      mediaType = mediaType.withCharset(config.getCharset());
    }

    return mediaType;
  }

  /**
   * Performs the request using the given {@code builder} and {@code connection} and completes
   * the {@code callback} accordingly, expecting the response to not contain a body.
   * <p>
   * If successful, the callback will be completed with an empty String as payload.
   *
   * @param connection            the connection to use
   * @param builder               the request builder
   * @param responseTimeoutMillis the request timeout in millis
   * @param streamingHelper       the {@link StreamingHelper}
   * @param callback              the operation's {@link CompletionCallback}
   */
  protected void doVoidRequest(RestConnection connection,
                               HttpRequestBuilder builder,
                               int responseTimeoutMillis,
                               StreamingHelper streamingHelper,
                               CompletionCallback<String, HttpResponseAttributes> callback) {
    try {
      connection.bodylessRequest(builder, responseTimeoutMillis, ANY, streamingHelper).whenComplete(handleResponse(callback));
    } catch (Throwable t) {
      callback.error(t);
    }
  }

  /**
   * Sets the request body, and media-type and content-length headers from it if those are not already defined.
   * This method should be called after all the other parameters for the request have already been set.
   *
   * @param builder               the request builder
   * @param body                  the request body (nullable)
   * @param streamingType         the streaming mode for this request
   */
  protected void setBody(HttpRequestBuilder builder, TypedValue<InputStream> body, StreamingType streamingType) {
    if (body != null) {
      byte[] bytes = null;

      if (streamingType.equals(ALWAYS)) {
        builder.removeHeader(CONTENT_LENGTH);
        builder.removeHeader(TRANSFER_ENCODING);
        builder.addHeader(TRANSFER_ENCODING, CHUNKED);
      } else if (streamingType.equals(NEVER)) {
        bytes = setNeverStreamingContentLength(body, builder);
      } else {
        setAutoContentLengthHeader(body, builder);
      }

      setContentType(body, builder);

      if (body.getValue() != null) {
        if (bytes != null) {
          builder.entity(new ByteArrayHttpEntity(bytes));
        } else {
          builder.entity(new InputStreamHttpEntity(body.getValue()));
        }
      }
    }
  }

  private <T> BiConsumer<Result<T, HttpResponseAttributes>, Throwable> handleResponse(
                                                                                      CompletionCallback<T, HttpResponseAttributes> callback) {

    return (r, e) -> {
      if (e != null) {
        callback.error(e);
      } else {
        callback.success(r);
      }
    };
  }

  /**
   * Allows to make a multipart request by facilitating the construction of the multipart body through a
   * {@link MultipartPayloadBuilder}.
   * <p>
   * This operation <b>MUST NOT</b> be completed using the {@code callback} passed as a parameter, but the one
   * provided through the {@code consumer}
   *
   * @param builderConfigurer a {@link Consumer} which gives access to the {@link MultipartPayloadBuilder} and allows to configure it
   * @param callback          the operation's {@link CompletionCallback}
   * @param consumer          the {@link Consumer} which gives access to the built multipart body and the effective callback that should
   *                          complete this operation
   * @param <T>               the generic type of the {@code callback} payload type
   * @param <A>               the generic type of the {@code callback} attributes type
   */
  protected <T, A> void withMultipart(Consumer<MultipartPayloadBuilder> builderConfigurer,
                                      CompletionCallback<T, A> callback,
                                      BiConsumer<TypedValue<InputStream>, CompletionCallback<T, A>> consumer) {

    MultipartPayloadBuilder builder = new DWMultipartPayloadBuilder(expressionLanguage);
    builderConfigurer.accept(builder);

    TypedValue<InputStream> multipart = builder.build();
    try {
      consumer.accept(multipart, new CloserCompletionCallbackDecorator<>(callback, multipart.getValue()));
    } catch (Exception e) {
      closeQuietly(multipart.getValue());
      callback.error(e);
    }
  }

  /**
   * Creates a preconfigured {@link HttpRequestBuilder} per the given parameters.
   * <p>
   * This creates a new {@link HttpRequestBuilder} and initializes it's URI, method, custom query params and custom headers.
   * <p>
   * The returned builder can still to add/change additional information.
   *
   * @param baseUri    the base uri to be used
   * @param path       the path of the resource to request, relative to the base URI
   * @param method     the request method
   * @param parameters the request parameters
   * @return a new {@link HttpRequestBuilder}
   */
  protected HttpRequestBuilder requestBuilder(String baseUri,
                                              String path,
                                              Method method,
                                              RequestParameters parameters) {

    final String uri = buildRequestUri(baseUri, path);
    HttpRequestBuilder builder = HttpRequest.builder().method(method).uri(uri);

    builder.headers(parameters.getCustomHeaders());
    builder.queryParams(parameters.getCustomQueryParams());

    return builder;
  }

  /**
   * This method specifies the {@link MediaType} that should be assumed the response to have in case the remote
   * service doesn't specify a {@code Content-Type} header.
   * <p>
   * By default, this method returns {@link MediaType#APPLICATION_JSON} since it's the most common response type. However,
   * this method should be overwritten if a different type of media type is to be expected, or if you know that a certain
   * encoding will be enforced.
   *
   * @return the {@link MediaType} to assign the response in case the server doesn't specify one in its response
   */
  protected MediaType getDefaultResponseMediaType() {
    return MediaType.APPLICATION_JSON;
  }

  private void setContentType(TypedValue<InputStream> body, HttpRequestBuilder builder) {
    if (!builder.getHeaderValue(CONTENT_TYPE).isPresent()) {
      MediaType mediaType = body.getDataType().getMediaType();
      if (mediaType != null && !mediaType.getPrimaryType().equals("*")) {
        builder.addHeader(CONTENT_TYPE, mediaType.toRfcString());
      }
    }
  }

  /**
   * Sets the content length header value for this requests, reading the body to get it if necessary.
   * If the body has been read this method returns it's bytes.
   * @param body        The request body
   * @param builder     The request builder
   * @return            A byte[] containing the body if the input stream has been read.
   */
  private byte[] setNeverStreamingContentLength(TypedValue<InputStream> body, HttpRequestBuilder builder) {
    Optional<String> customLength = builder.getHeaderValue(CONTENT_LENGTH);

    byte[] bytes = null;

    if (!customLength.isPresent()) {
      if (body.getByteLength().isPresent()) {
        builder.addHeader(CONTENT_LENGTH, valueOf(body.getByteLength().getAsLong()));
      } else if (body.getValue() != null) {
        bytes = toByteArray(body.getValue());
        builder.addHeader(CONTENT_LENGTH, valueOf(bytes.length));
      }
    }

    builder.removeHeader(TRANSFER_ENCODING);

    return bytes;
  }

  private void setAutoContentLengthHeader(TypedValue<InputStream> body, HttpRequestBuilder builder) {
    Optional<String> customLength = builder.getHeaderValue(CONTENT_LENGTH);
    boolean isChunked = builder.getHeaderValue(TRANSFER_ENCODING).map(CHUNKED::equals).orElse(false);

    if (body.getByteLength().isPresent()) {
      boolean addHeader = true;
      String length = valueOf(body.getByteLength().getAsLong());
      if (customLength.isPresent()) {
        LOGGER.warn("Invoking URI {} with body of known length {}. However, a {} header with value {} was manually specified. "
            + "Will proceed with the custom value.",
                    builder.getUri(), length, CONTENT_LENGTH, customLength.get());

        addHeader = false;
      }

      if (isChunked) {
        LOGGER.debug("Invoking URI {} with a manually set {}: {} header, even though body is of known length {}. "
            + "Skipping automatic addition of {} header", builder.getUri(), TRANSFER_ENCODING, CHUNKED, length, CONTENT_LENGTH);
        addHeader = false;
      }

      if (addHeader) {
        builder.addHeader(CONTENT_LENGTH, length);
      }

    } else if (!isChunked && !customLength.isPresent()) {
      builder.addHeader(TRANSFER_ENCODING, CHUNKED);
    }
  }

  private String buildRequestUri(String baseUri, String path) {
    boolean pathStartsWithSlash = path.startsWith("/");
    boolean baseEndsInSlash = baseUri.endsWith("/");

    if (pathStartsWithSlash && baseEndsInSlash) {
      path = path.substring(1);
    } else if (!pathStartsWithSlash && !baseEndsInSlash) {
      baseUri += '/';
    }

    return baseUri + path;
  }
}
