/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.connectors.restconnect.commons.api.operation.paging;

import static org.mule.connectors.restconnect.commons.internal.util.RestConnectUtils.isBlank;
import static org.mule.runtime.api.metadata.DataType.STRING;

import org.mule.connectors.restconnect.commons.api.connection.RestConnection;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.mule.runtime.http.api.domain.message.request.HttpRequestBuilder;

import java.util.List;
import java.util.function.Function;

/**
 * Implementation of {@link RestPagingProvider} for APIs which do paging based on a marker provided in the response of each
 * request, acting as a marker or position inside the stream of results.
 * <p>
 * Request number 1 will obtain the first page plus a marker embedded in the response. Said marker will be added as a
 * query param in request number 2, which will reply with the next page and a new marker which overwrites the prior one. The
 * process continues until no new marker is provided or the next page comes back empty.
 *
 * @since 1.0
 */
public class MarkerPagingProvider extends RestPagingProvider {

  private final String markerParamName;
  private final String nextMarkerExpression;

  private boolean firstPage = true;
  private String nextMarker;

  /**
   * Creates a new instance
   *
   * @param markerParamName      the name of the query parameter in which the marker is placed in the request
   * @param nextMarkerExpression the DW expression used to extract the next marker from the response
   * @param requestFactory       a {@link Function} to generate the request to be used on each page request. Each invocation should yield a different instance
   * @param expressionLanguage   the app's {@link ExpressionLanguage}
   * @param streamingHelper      the {@link StreamingHelper} associated to the executing operation
   * @param payloadExpression    a DW expression to extract the data from the response
   * @param defaultMediaType     the {@link MediaType} for the page items if the server doesn't specify one
   * @param responseTimeout      the timeout for each request
   */
  public MarkerPagingProvider(String markerParamName,
                              String nextMarkerExpression,
                              Function<RestConnection, HttpRequestBuilder> requestFactory,
                              ExpressionLanguage expressionLanguage,
                              StreamingHelper streamingHelper,
                              String payloadExpression,
                              MediaType defaultMediaType,
                              int responseTimeout) {
    super(requestFactory, expressionLanguage, streamingHelper, payloadExpression, defaultMediaType, responseTimeout);
    this.markerParamName = markerParamName;
    this.nextMarkerExpression = nextMarkerExpression;
  }

  @Override
  protected void configureRequest(HttpRequestBuilder requestBuilder) {
    if (!firstPage) {
      if (isBlank(nextMarker)) {
        stopPaging();
        return;
      }

      requestBuilder.addQueryParam(markerParamName, nextMarker);
    }
  }

  @Override
  protected void onPage(List<TypedValue<String>> page, TypedValue<String> rawPage) {
    firstPage = false;

    if (isBlank(rawPage.getValue())) {
      nextMarker = null;
      stopPaging();
    } else {
      nextMarker = (String) evaluate(rawPage, nextMarkerExpression, STRING).getValue();
    }
  }
}
