/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.connectors.restconnect.commons.internal.util;

import static org.mule.metadata.xml.api.SchemaCollector.getInstance;
import static org.mule.runtime.core.api.functional.Either.left;
import static org.mule.runtime.core.api.functional.Either.right;
import static org.mule.runtime.core.api.util.IOUtils.closeQuietly;

import org.mule.connectors.restconnect.commons.internal.datasense.loader.CustomJsonSchemaTypeLoader;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.xml.api.SchemaCollector;
import org.mule.metadata.xml.api.XmlTypeLoader;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.CursorProvider;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.runtime.core.api.functional.Either;
import org.mule.runtime.core.api.util.IOUtils;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.Optional;

/**
 * Utility methods
 *
 * @since 1.0
 */
public final class RestConnectUtils {

  private RestConnectUtils() {}

  /**
   * Closes the given {@code cursorProvider} and releases all associated resources
   *
   * @param cursorProvider a cursor provider
   */
  public static void closeAndRelease(CursorProvider cursorProvider) {
    cursorProvider.close();
    cursorProvider.releaseResources();
  }

  /***
   * Closes the given {@code stream} which can be either a {@link CursorProvider} or an {@link InputStream}.
   *
   * Null values or instances of other classes will be ignored.
   *
   * @param stream a stream
   */
  public static void closeStream(Object stream) {
    if (stream instanceof CursorProvider) {
      closeAndRelease((CursorProvider) stream);
    } else if (stream instanceof InputStream) {
      closeQuietly((InputStream) stream);
    }
  }

  /**
   * Consumes the contents of the given {@code stream} as a String and closes it.
   * <p>
   * The {@code stream} can be either a {@link CursorStreamProvider} or an {@link InputStream}.
   * {@link IllegalArgumentException} will be thrown if the {@code stream} is an instance of any other class.
   * <p>
   * Notice that in the case of {@link CursorStreamProvider}, this method will close the provider meaning that it will no longer
   * be able to yield new cursors.
   *
   * @param stream   a {@link CursorStreamProvider} or an {@link InputStream}
   * @param targetCharset the encoding to use when reading the String and that will be set to the TypedValue
   * @param targetMediaType the media type that will be set to the TypedValue
   * @return the contents as a String
   * @throws IllegalArgumentException if {@code stream} is not of the expected types.
   */
  public static TypedValue<String> consumeStringAndClose(Object stream, MediaType targetMediaType, Charset targetCharset) {
    return consumeStringTransformAndClose(stream, targetCharset, targetMediaType, targetCharset);
  }


  /**
   * Consumes the contents of the given {@code stream} as a String and closes it.
   * <p>
   * The {@code stream} can be either a {@link CursorStreamProvider} or an {@link InputStream}.
   * {@link IllegalArgumentException} will be thrown if the {@code stream} is an instance of any other class.
   * <p>
   * Notice that in the case of {@link CursorStreamProvider}, this method will close the provider meaning that it will no longer
   * be able to yield new cursors.
   *
   * @param stream   a {@link CursorStreamProvider} or an {@link InputStream}
   * @param sourceCharset the encoding to use when reading the String
   * @param targetCharset the encoding that will be set to the TypedValue
   * @param targetMediaType the media type that will be set to the TypedValue
   * @return the contents as a String
   * @throws IllegalArgumentException if {@code stream} is not of the expected types.
   */
  public static TypedValue<String> consumeStringTransformAndClose(Object stream, Charset sourceCharset, MediaType targetMediaType,
                                                                  Charset targetCharset) {
    if (stream == null) {
      return toTypedValue("", targetMediaType, targetCharset);
    }

    if (stream instanceof String) {
      return toTypedValue((String) stream, targetMediaType, targetCharset);
    }

    Either<CursorStreamProvider, InputStream> content;

    if (stream instanceof CursorStreamProvider) {
      content = left((CursorStreamProvider) stream);
    } else if (stream instanceof InputStream) {
      content = right((InputStream) stream);
    } else {
      throw new IllegalArgumentException("Cannot consume stream of unsupported type: " + stream.getClass().getName());
    }

    return content.reduce(provider -> {
      try {
        return doConsumeAndClose(provider.openCursor(), sourceCharset, targetMediaType, targetCharset);
      } finally {
        closeAndRelease(provider);
      }
    }, in -> doConsumeAndClose(in, sourceCharset, targetMediaType, targetCharset));
  }

  private static TypedValue<String> doConsumeAndClose(InputStream stream, Charset sourceCharset, MediaType targetMediaType,
                                                      Charset targetCharset) {
    try {
      return toTypedValue(IOUtils.toString(stream, sourceCharset), targetMediaType, targetCharset);
    } finally {
      closeQuietly(stream);
    }
  }

  private static TypedValue<String> toTypedValue(String value, MediaType mediaType, Charset encoding) {
    return new TypedValue<>(value, DataType.builder().mediaType(mediaType).charset(encoding).build());
  }

  public static boolean isBlank(String value) {
    return value == null || value.trim().length() == 0;
  }

  public static boolean containsIgnoreCase(String value, String predicate) {
    if (value == null || predicate == null) {
      return false;
    }

    return value.toLowerCase().contains(predicate.toLowerCase());
  }

  public static boolean isNotBlank(String v) {
    return v != null && v.trim().length() > 0;
  }

  public static String readSchema(ClassLoader classLoader, String schemaPath) {
    String schema;

    if (schemaPath.startsWith("/")) {
      schemaPath = schemaPath.substring(1);
    }

    try (InputStream in = classLoader.getResourceAsStream(schemaPath)) {
      if (in == null) {
        throw new IllegalArgumentException("Could not find schema at " + schemaPath);
      }

      schema = IOUtils.toString(in);
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
    return schema;
  }

  public static MetadataType loadXmlSchema(String schemaPath, String schemaContent, String qName) {
    SchemaCollector schemaCollector = getInstance().addSchema(schemaPath, new ByteArrayInputStream(schemaContent.getBytes()));

    Optional<MetadataType> metadataType = new XmlTypeLoader(schemaCollector).load(qName);

    if (!metadataType.isPresent()) {
      throw new RuntimeException("Could not load XML Schema " + schemaPath + " QName:" + qName);
    }

    return metadataType.get();
  }

  public static MetadataType loadJsonSchema(String schemaPath, String schemaContent, MetadataFormat metadataFormat) {

    Optional<MetadataType> metadataType = new CustomJsonSchemaTypeLoader(schemaContent, metadataFormat).load(null);

    if (!metadataType.isPresent()) {
      throw new RuntimeException("Could not load Json Schema " + schemaPath);
    }

    return metadataType.get();
  }
}
