/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.connectors.restconnect.commons.internal.datasense.loader;

import org.everit.json.schema.Schema;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.mule.metadata.api.TypeLoader;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.utils.MetadataTypeUtils;
import org.mule.metadata.json.api.handler.ParsingContext;

import java.util.Optional;

import static org.mule.metadata.json.api.utils.SchemaHelper.generateSchema;

/***
 * This class loads the metadata structure from a JSONSchema as JsonTypeLoader would do but allows
 * defining a MetadataFormat for it instead of setting it to JSON.
 *
 * This class is needed for Rest Connect Commons to be backwards compatible with Mule 4.2.1
 *
 * @since 1.0
 */
public class CustomJsonSchemaTypeLoader implements TypeLoader {

  private JsonSchemaStringLoader jsonSchemaLoader;
  private MetadataFormat metadataType;

  public CustomJsonSchemaTypeLoader(String schemaData, MetadataFormat metadataType) {
    this.metadataType = metadataType;
    jsonSchemaLoader = new JsonSchemaStringLoader(schemaData);
  }

  @Override
  public Optional<MetadataType> load(String identifier) {
    final Schema jsonSchema = jsonSchemaLoader.loadSchema();
    final TypeBuilder<?> typeBuilder =
        new CustomMetadataFormatHandlerManager(metadataType).handle(jsonSchema, new ParsingContext());
    MetadataTypeUtils.addTypeAlias(typeBuilder, null);
    return Optional.of(typeBuilder.build());
  }

  private static class JsonSchemaStringLoader {

    private String schemaData;

    public JsonSchemaStringLoader(String schemaData) {
      this.schemaData = schemaData;
    }

    public Schema loadSchema() {
      final JSONObject rawSchema = new JSONObject(new JSONTokener(schemaData));
      return generateSchema(rawSchema, null);
    }
  }
}
