/*
 * Copyright © MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.connectors.restconnect.commons.api.error;

import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JAVA;
import org.mule.connectors.restconnect.commons.api.operation.HttpResponseAttributes;
import org.mule.runtime.api.exception.ErrorMessageAwareException;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.api.runtime.operation.Result;

/**
 * {@link ModuleException} to be raised when a request fails.
 * <p>
 * Because the HTTP protocol also emits responses when the request fails, this exception carries the
 * error response by implementing the {@link ErrorMessageAwareException}. The response is accessible through
 * the {@link #getErrorMessage()} method.
 *
 * @since 1.0
 */
public class RequestException extends ModuleException implements ErrorMessageAwareException {

  private final Message message;

  /**
   * Creates a new instance
   *
   * @param error    the error type
   * @param response the obtained response
   */
  public RequestException(RestConnectError error, Result<Object, HttpResponseAttributes> response) {
    super("Request returned status code " + response.getAttributes().get().getStatusCode(), error);
    Message.Builder builder = Message.builder()
        .payload(getPayloadTypedValue(response))
        .mediaType(getMediaType(response));

    setAttributes(response, builder);
    message = builder.build();
  }


  private TypedValue<Object> getPayloadTypedValue(Result<Object, HttpResponseAttributes> response) {
    DataType dataType = DataType.builder()
        .type(response.getOutput().getClass())
        .mediaType(getMediaType(response))
        .build();

    return new TypedValue<>(response.getOutput(), dataType, response.getByteLength());
  }

  private void setAttributes(Result<Object, HttpResponseAttributes> response, Message.Builder builder) {
    response.getAttributes().ifPresent(attributes -> {
      DataType dataType = DataType.builder()
          .type(attributes.getClass())
          .mediaType(response.getAttributesMediaType().orElse(APPLICATION_JAVA))
          .build();

      builder.attributes(new TypedValue<>(attributes, dataType));
    });

  }

  private MediaType getMediaType(Result<Object, HttpResponseAttributes> response) {
    return response.getMediaType().orElse(null);
  }

  @Override
  public Message getErrorMessage() {
    return message;
  }
}
