package org.mule.soap.api.security;

import com.google.common.collect.ImmutableMap;
import org.mule.soap.internal.security.callback.WSPasswordCallbackHandler;

import java.util.Map;
import java.util.Optional;

import static java.util.Optional.empty;
import static org.apache.wss4j.common.ConfigurationConstants.TIMESTAMP;
import static org.apache.wss4j.common.ConfigurationConstants.TTL_TIMESTAMP;

/**
 * Bundles the outgoing SOAP message that it's being built with a timestamp that carries the creation.
 *
 * @since 1.0
 */
public class WssTimestampSecurityStrategy implements SecurityStrategy {

  /**
   * The time difference between creation and expiry time in seconds. After this time the message is invalid.
   */
  private long timeToLiveInSeconds;

  /**
   * If true, the difference between creation and expiry time is verified in the response.
   * <p>
   * You should set this attribute to true if the server sends a timestamp as a response to a timestamp header in the security
   * header
   */
  private boolean checkResponseTimestamp;

  public WssTimestampSecurityStrategy(long timeToLeaveInSeconds, boolean checkResponseTimestamp) {
    this.timeToLiveInSeconds = timeToLeaveInSeconds;
    this.checkResponseTimestamp = checkResponseTimestamp;
  }

  public WssTimestampSecurityStrategy(long timeToLeaveInSeconds) {
    this(timeToLeaveInSeconds, false);
  }

  @Override
  public SecurityStrategyType securityType() {
    return checkResponseTimestamp ? SecurityStrategyType.ALWAYS : SecurityStrategyType.OUTGOING;
  }

  @Override
  public Optional<WSPasswordCallbackHandler> buildPasswordCallbackHandler() {
    return empty();
  }

  @Override
  public String securityAction() {
    return TIMESTAMP;
  }

  @Override
  public Map<String, Object> buildSecurityProperties() {
    return ImmutableMap.<String, Object>builder().put(TTL_TIMESTAMP, String.valueOf(timeToLiveInSeconds)).build();
  }
}
