/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.generator.attachment;

import org.apache.cxf.message.Exchange;
import org.mule.metadata.api.TypeLoader;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.NullType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.soap.api.exception.InvalidWsdlException;
import org.mule.soap.internal.util.SoapServiceMetadataTypeUtils;
import org.mule.soap.internal.xml.util.XMLUtils;
import org.mule.wsdl.parser.model.operation.OperationModel;
import org.w3c.dom.Document;

import javax.wsdl.Part;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import static java.lang.String.format;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toMap;

/**
 * Abstract implementation for a request enricher that extracts the attachments from a SOAP Response.
 *
 * @since 1.0
 */
public abstract class AttachmentResponseEnricher {

  private final Map<String, OperationModel> operations;

  AttachmentResponseEnricher(List<OperationModel> operations) {
    this.operations = operations.stream().collect(toMap(OperationModel::getName, identity()));
  }

  /**
   * {@inheritDoc}
   * <p>
   * Modifies the SOAP response to avoid attachment content in the response body and make decouple the attachment handling so
   * the user can have a better experience.
   */
  public String enrich(Document response, String operation, Exchange exchange) {
    MetadataType attachments = operations.get(operation).getOutputType().getAttachments();

    if (attachments instanceof ObjectType) {
      Collection<ObjectFieldType> attachmentsFields = ((ObjectType) attachments).getFields();
      if (!attachmentsFields.isEmpty()) {
        processResponseAttachments(response, attachmentsFields, exchange);
      }
    }
    return XMLUtils.toXml(response);
  }

  /**
   * Processes the attachments nodes in the response.
   */
  protected abstract void processResponseAttachments(Document response, Collection<ObjectFieldType> attachments,
                                                     Exchange exchange);
}
