/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.generator;

import org.apache.cxf.message.Exchange;
import org.mule.soap.api.client.BadResponseException;
import org.mule.soap.api.message.SoapAttachment;
import org.mule.soap.api.message.SoapResponse;
import org.mule.soap.internal.client.AbstractSoapCxfClient;
import org.mule.soap.internal.generator.attachment.AttachmentResponseEnricher;
import org.mule.soap.internal.message.DefaultSoapResponse;
import org.mule.soap.internal.message.EmptySoapResponse;
import org.mule.soap.internal.util.XmlTransformationException;
import org.mule.soap.internal.util.XmlTransformationUtils;
import org.w3c.dom.Document;

import javax.xml.stream.XMLStreamReader;
import java.io.ByteArrayInputStream;
import java.util.Map;

import static java.util.Collections.emptyMap;
import static org.mule.soap.internal.client.AbstractSoapCxfClient.MULE_ATTACHMENTS_KEY;
import static org.mule.soap.internal.client.AbstractSoapCxfClient.MULE_HEADERS_KEY;

/**
 *
 * @since 1.0
 */
public final class SoapResponseGenerator {

  private final AttachmentResponseEnricher responseEnricher;

  public SoapResponseGenerator(AttachmentResponseEnricher responseEnricher) {
    this.responseEnricher = responseEnricher;
  }

  /**
   *
   * @param operation the name of the operation that was invoked
   * @param response  the CXF response
   * @param exchange  the exchange used for CXF to store the headers and attachments.
   */
  public SoapResponse generate(String operation, Object[] response, Exchange exchange) {
    Map<String, String> transportHeaders = getTransportHeaders(exchange);
    Map<String, String> transportAdditionalData = getTransportAdditionalData(exchange);
    if (response == null) {
      return new EmptySoapResponse(transportHeaders, transportAdditionalData);
    }

    Document document = unwrapResponse(response);
    String result = responseEnricher.enrich(document, operation, exchange);
    Map<String, SoapAttachment> attachments = (Map<String, SoapAttachment>) exchange.get(MULE_ATTACHMENTS_KEY);
    Map<String, String> headers = (Map<String, String>) exchange.get(MULE_HEADERS_KEY);
    ByteArrayInputStream body = new ByteArrayInputStream(result.getBytes());
    return new DefaultSoapResponse(body, headers, transportHeaders, transportAdditionalData, attachments, "application/xml");
  }

  private Map<String, String> getTransportHeaders(Exchange exchange) {
    Map<String, String> transportHeaders = (Map<String, String>) exchange.get(AbstractSoapCxfClient.MULE_TRANSPORT_HEADERS_KEY);
    return transportHeaders == null ? emptyMap() : transportHeaders;
  }

  private Map<String, String> getTransportAdditionalData(Exchange exchange) {
    Map<String, String> transportAdditionalData =
        (Map<String, String>) exchange.get(AbstractSoapCxfClient.MULE_TRANSPORT_ADDITIONAL_DATA_KEY);
    return transportAdditionalData == null ? emptyMap() : transportAdditionalData;
  }

  /**
   * Unwraps the CXF {@link XMLStreamReader} response into a dom {@link Document}.
   *
   * @param response the CXF received response.
   */
  private Document unwrapResponse(Object[] response) {
    if (response.length == 0) {
      throw new BadResponseException("no elements were received in the SOAP response.");
    }
    if (response.length != 1) {
      throw new BadResponseException("the obtained response contains more than one element, only one was expected");
    }
    XMLStreamReader reader = (XMLStreamReader) response[0];
    try {
      return XmlTransformationUtils.xmlStreamReaderToDocument(reader);
    } catch (XmlTransformationException e) {
      throw new BadResponseException("Error transforming the XML web service response to be processed", e);
    }
  }
}
