/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.client;

import com.google.common.collect.ImmutableMap;
import org.apache.cxf.attachment.AttachmentImpl;
import org.apache.cxf.endpoint.Client;
import org.apache.cxf.message.Attachment;
import org.mule.metadata.api.TypeLoader;
import org.mule.soap.api.client.SoapClient;
import org.mule.soap.api.message.SoapAttachment;
import org.mule.soap.internal.generator.attachment.MtomRequestEnricher;
import org.mule.soap.internal.generator.attachment.MtomResponseEnricher;
import org.mule.wsdl.parser.model.PortModel;
import org.mule.wsdl.parser.model.WsdlModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.activation.DataHandler;
import java.util.Map;

import static com.google.common.net.HttpHeaders.CONTENT_DISPOSITION;

/**
 * a {@link SoapClient} implementation based on CXF for MTOM web services.
 *
 * @since 1.0
 */
public class SoapMtomCxfClient extends AbstractSoapCxfClient {

  SoapMtomCxfClient(Client client,
                    WsdlModel wsdlModel,
                    PortModel portModel,
                    String address,
                    String encoding) {
    super(client, wsdlModel, portModel, address, encoding, new MtomRequestEnricher(),
          new MtomResponseEnricher(portModel.getOperations()));
  }

  @Override
  protected Map<String, Attachment> buildCxfAttachments(Map<String, SoapAttachment> attachments) {
    ImmutableMap.Builder<String, Attachment> builder = ImmutableMap.builder();
    attachments.forEach((name, value) -> {
      AttachmentImpl attachment =
          new AttachmentImpl(name, new DataHandler(new StreamDataSource(value.getContent(), value.getContentType(), name)));
      attachment.setHeader(CONTENT_DISPOSITION, "attachment; name=\"" + name + "\"");
      builder.put(name, attachment);
    });
    return builder.build();
  }
}
